/**
 * Administration JavaScript - Portail Vert
 * Gestion des gestionnaires
 */

let currentManagerId = null;
let isEditMode = false;

// Vérification de l'authentification au chargement de la page
document.addEventListener('DOMContentLoaded', function() {
    // Vérifier si l'utilisateur est authentifié
    if (sessionStorage.getItem('authenticated') !== 'true') {
        window.location.href = '/login.html';
        return;
    }
    
    // Initialiser l'administration
    initializeAdmin();
});

/**
 * Initialise l'interface d'administration
 */
function initializeAdmin() {
    loadManagers();
    setupEventListeners();
}

/**
 * Configure tous les événements
 */
function setupEventListeners() {
    // Boutons de navigation
    document.getElementById('backToDashboard').addEventListener('click', () => {
        window.location.href = '/dashboard.html';
    });
    
    document.getElementById('logoutBtn').addEventListener('click', logout);
    
    // Bouton d'ajout de gestionnaire
    document.getElementById('addManagerBtn').addEventListener('click', showAddManagerModal);
    
    // Modal de gestionnaire
    document.getElementById('managerForm').addEventListener('submit', saveManager);
    document.getElementById('cancelManagerBtn').addEventListener('click', hideManagerModal);
    
    // Modal de suppression
    document.getElementById('confirmDelete').addEventListener('click', deleteManager);
    document.getElementById('cancelDelete').addEventListener('click', hideDeleteModal);
}

/**
 * Charge la liste des gestionnaires
 */
async function loadManagers() {
    const loadingDiv = document.getElementById('managersLoading');
    const listDiv = document.getElementById('managersList');
    const emptyDiv = document.getElementById('noManagers');
    
    loadingDiv.style.display = 'block';
    listDiv.style.display = 'none';
    emptyDiv.style.display = 'none';
    
    try {
        const response = await fetch('/admin/managers');
        
        if (response.status === 403) {
            showNotification('Accès administrateur requis', 'error');
            window.location.href = '/dashboard.html';
            return;
        }
        
        const result = await response.json();
        
        if (response.ok) {
            displayManagers(result.managers);
        } else {
            console.error('Erreur lors du chargement des gestionnaires:', result.error);
            showNotification('Erreur lors du chargement des gestionnaires', 'error');
        }
    } catch (error) {
        console.error('Erreur lors du chargement des gestionnaires:', error);
        showNotification('Erreur de connexion au serveur', 'error');
    } finally {
        loadingDiv.style.display = 'none';
    }
}

/**
 * Affiche la liste des gestionnaires
 */
function displayManagers(managers) {
    const listDiv = document.getElementById('managersList');
    const emptyDiv = document.getElementById('noManagers');
    
    if (managers.length === 0) {
        listDiv.style.display = 'none';
        emptyDiv.style.display = 'block';
        return;
    }
    
    listDiv.innerHTML = '';
    listDiv.style.display = 'block';
    emptyDiv.style.display = 'none';
    
    managers.forEach(manager => {
        const managerElement = createManagerElement(manager);
        listDiv.appendChild(managerElement);
    });
}

/**
 * Crée un élément de gestionnaire
 */
function createManagerElement(manager) {
    const div = document.createElement('div');
    div.className = 'manager-item';
    div.dataset.managerId = manager.id;
    
    const roleText = manager.role === 'admin' ? '👑 Administrateur' : '👤 Gestionnaire';
    const lastLogin = manager.lastLogin ? 
        new Date(manager.lastLogin).toLocaleString('fr-FR') : 
        'Jamais connecté';
    
    // Statut 2FA
    let twoFAStatus = '';
    let twoFAClass = '';
    if (manager.isFirstLogin) {
        twoFAStatus = '⚠️ Configuration requise';
        twoFAClass = 'pending';
    } else if (manager.totpEnabled) {
        twoFAStatus = '🔒 2FA activé';
        twoFAClass = 'enabled';
    } else {
        twoFAStatus = '❌ 2FA désactivé';
        twoFAClass = 'disabled';
    }
    
    div.innerHTML = `
        <div class="manager-header">
            <div class="manager-info">
                <h3>${escapeHtml(manager.name)}</h3>
                <p class="manager-role">${roleText}</p>
                <span class="manager-2fa-status ${twoFAClass}">${twoFAStatus}</span>
            </div>
            <div class="manager-actions">
                <button class="btn-secondary btn-small" onclick="editManager('${manager.id}')">
                    ✏️ Modifier
                </button>
                <button class="btn-reset-2fa btn-small" onclick="resetManagerTOTP('${manager.id}', '${escapeHtml(manager.name)}')" 
                        title="Réinitialiser le 2FA">
                    🔄 Reset 2FA
                </button>
                <button class="btn-danger btn-small" onclick="confirmDeleteManager('${manager.id}', '${escapeHtml(manager.name)}')">
                    🗑️ Supprimer
                </button>
            </div>
        </div>
        <div class="manager-details">
            <span><strong>N° employé:</strong> ${escapeHtml(manager.employeeNumber)}</span>
            <span><strong>Créé le:</strong> ${new Date(manager.createdAt).toLocaleString('fr-FR')}</span>
            <span><strong>Dernière connexion:</strong> ${lastLogin}</span>
        </div>
    `;
    
    return div;
}

/**
 * Affiche la modal d'ajout de gestionnaire
 */
function showAddManagerModal() {
    isEditMode = false;
    currentManagerId = null;
    
    document.getElementById('modalTitle').textContent = 'Ajouter un gestionnaire';
    document.getElementById('managerForm').reset();
    document.getElementById('managerPassword').required = true;
    document.getElementById('saveManagerBtn').querySelector('.btn-text').textContent = 'Ajouter';
    
    document.getElementById('managerModal').style.display = 'flex';
}

/**
 * Affiche la modal de modification de gestionnaire
 */
async function editManager(managerId) {
    try {
        const response = await fetch('/admin/managers');
        const result = await response.json();
        
        if (response.ok) {
            const manager = result.managers.find(m => m.id === managerId);
            if (manager) {
                isEditMode = true;
                currentManagerId = managerId;
                
                document.getElementById('modalTitle').textContent = 'Modifier le gestionnaire';
                document.getElementById('managerName').value = manager.name;
                document.getElementById('employeeNumber').value = manager.employeeNumber;
                document.getElementById('managerPassword').value = '';
                document.getElementById('managerPassword').required = false;
                document.getElementById('managerRole').value = manager.role;
                document.getElementById('saveManagerBtn').querySelector('.btn-text').textContent = 'Modifier';
                
                document.getElementById('managerModal').style.display = 'flex';
            }
        }
    } catch (error) {
        console.error('Erreur lors du chargement du gestionnaire:', error);
        showNotification('Erreur lors du chargement du gestionnaire', 'error');
    }
}

/**
 * Cache la modal de gestionnaire
 */
function hideManagerModal() {
    document.getElementById('managerModal').style.display = 'none';
    document.getElementById('managerError').style.display = 'none';
}

/**
 * Sauvegarde un gestionnaire
 */
async function saveManager(e) {
    e.preventDefault();
    
    const saveBtn = document.getElementById('saveManagerBtn');
    const btnText = saveBtn.querySelector('.btn-text');
    const btnLoading = saveBtn.querySelector('.btn-loading');
    const errorDiv = document.getElementById('managerError');
    
    // Afficher l'état de chargement
    btnText.style.display = 'none';
    btnLoading.style.display = 'inline';
    saveBtn.disabled = true;
    errorDiv.style.display = 'none';
    
    const formData = new FormData(e.target);
    const managerData = {
        name: formData.get('name'),
        employeeNumber: formData.get('employeeNumber'),
        password: formData.get('password'),
        role: formData.get('role')
    };
    
    try {
        let url = '/admin/managers';
        let method = 'POST';
        
        if (isEditMode) {
            url = `/admin/managers/${currentManagerId}`;
            method = 'PUT';
            
            // Ne pas envoyer le mot de passe s'il est vide en mode édition
            if (!managerData.password) {
                delete managerData.password;
            }
        }
        
        const response = await fetch(url, {
            method: method,
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(managerData)
        });
        
        const result = await response.json();
        
        if (response.ok && result.success) {
            hideManagerModal();
            loadManagers();
            showNotification(
                isEditMode ? 'Gestionnaire modifié avec succès' : 'Gestionnaire ajouté avec succès', 
                'success'
            );
        } else {
            errorDiv.textContent = result.error || 'Erreur lors de la sauvegarde';
            errorDiv.style.display = 'block';
        }
    } catch (error) {
        console.error('Erreur lors de la sauvegarde:', error);
        errorDiv.textContent = 'Erreur de connexion au serveur';
        errorDiv.style.display = 'block';
    } finally {
        // Restaurer l'état du bouton
        btnText.style.display = 'inline';
        btnLoading.style.display = 'none';
        saveBtn.disabled = false;
    }
}

/**
 * Confirme la suppression d'un gestionnaire
 */
function confirmDeleteManager(managerId, managerName) {
    currentManagerId = managerId;
    document.getElementById('deleteMessage').textContent = 
        `Êtes-vous sûr de vouloir supprimer le gestionnaire "${managerName}" ?`;
    document.getElementById('deleteModal').style.display = 'flex';
}

/**
 * Cache la modal de suppression
 */
function hideDeleteModal() {
    document.getElementById('deleteModal').style.display = 'none';
    currentManagerId = null;
}

/**
 * Supprime un gestionnaire
 */
async function deleteManager() {
    if (!currentManagerId) return;
    
    try {
        const response = await fetch(`/admin/managers/${currentManagerId}`, {
            method: 'DELETE'
        });
        
        const result = await response.json();
        
        if (response.ok && result.success) {
            hideDeleteModal();
            loadManagers();
            showNotification('Gestionnaire supprimé avec succès', 'success');
        } else {
            showNotification(result.error || 'Erreur lors de la suppression', 'error');
        }
    } catch (error) {
        console.error('Erreur lors de la suppression:', error);
        showNotification('Erreur de connexion au serveur', 'error');
    }
}

/**
 * Déconnecte l'utilisateur
 */
async function logout() {
    try {
        await fetch('/api/logout', { method: 'POST' });
    } catch (error) {
        console.error('Erreur lors de la déconnexion:', error);
    } finally {
        sessionStorage.removeItem('authenticated');
        window.location.href = '/login.html';
    }
}

/**
 * Affiche une notification
 */
function showNotification(message, type = 'info') {
    // Réutiliser la fonction du dashboard
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.textContent = message;
    
    Object.assign(notification.style, {
        position: 'fixed',
        top: '20px',
        right: '20px',
        padding: '15px 20px',
        borderRadius: '8px',
        color: 'white',
        fontWeight: '500',
        zIndex: '9999',
        maxWidth: '300px',
        boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
        transform: 'translateX(100%)',
        transition: 'transform 0.3s ease-out'
    });
    
    const colors = {
        success: '#28a745',
        error: '#dc3545',
        warning: '#ffc107',
        info: '#17a2b8'
    };
    
    notification.style.backgroundColor = colors[type] || colors.info;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.style.transform = 'translateX(0)';
    }, 100);
    
    setTimeout(() => {
        notification.style.transform = 'translateX(100%)';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }, 5000);
}

/**
 * Réinitialise le TOTP d'un gestionnaire
 */
async function resetManagerTOTP(managerId, managerName) {
    if (!confirm(`Êtes-vous sûr de vouloir réinitialiser le 2FA pour "${managerName}" ?\n\nCette action obligera l'utilisateur à reconfigurer son authentification à deux facteurs lors de sa prochaine connexion.`)) {
        return;
    }

    try {
        const response = await fetch(`/admin/managers/${managerId}/reset-totp`, {
            method: 'POST'
        });

        const result = await response.json();

        if (response.ok && result.success) {
            showNotification(result.message, 'success');
            loadManagers(); // Recharger la liste pour refléter les changements
        } else {
            showNotification(result.error || 'Erreur lors de la réinitialisation du 2FA', 'error');
        }
    } catch (error) {
        console.error('Erreur lors de la réinitialisation TOTP:', error);
        showNotification('Erreur de connexion au serveur', 'error');
    }
}

/**
 * Échappe les caractères HTML
 */
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}
