// Variables globales
let sessionId = null;
let archiveDate = null;

/**
 * Initialise la vue d'archive au chargement de la page
 */
document.addEventListener('DOMContentLoaded', function() {
    // Récupérer les paramètres de l'URL
    const urlParams = new URLSearchParams(window.location.search);
    sessionId = urlParams.get('session');
    archiveDate = urlParams.get('date');

    if (!sessionId || !archiveDate) {
        showError('Paramètres manquants dans l\'URL');
        return;
    }

    // Configurer les événements
    setupEventListeners();

    // Charger la conversation archivée
    loadArchivedConversation();
});

/**
 * Configure les événements de la page
 */
function setupEventListeners() {
    // Boutons de retour au dashboard
    const backButtons = [
        document.getElementById('backToDashboard'),
        document.getElementById('backToDashboardBtn')
    ];
    
    backButtons.forEach(btn => {
        if (btn) {
            btn.addEventListener('click', () => {
                window.location.href = '/dashboard.html';
            });
        }
    });

    // Bouton d'impression
    const printBtn = document.getElementById('printConversationBtn');
    if (printBtn) {
        printBtn.addEventListener('click', () => {
            window.print();
        });
    }
}

/**
 * Charge la conversation archivée
 */
async function loadArchivedConversation() {
    showLoadingIndicator(true);

    try {
        const response = await fetch(`/portal/archives?date=${archiveDate}`);
        const conversations = await response.json();

        if (response.ok) {
            // Trouver la conversation spécifique
            const conversation = conversations.find(conv => conv.id === sessionId);
            
            if (conversation) {
                // Maintenant charger les détails complets depuis le fichier d'archive
                await loadConversationDetails(conversation);
            } else {
                showError('Conversation non trouvée dans les archives de cette date');
            }
        } else {
            showError(conversations.error || 'Erreur lors du chargement des archives');
        }
    } catch (error) {
        console.error('Erreur lors du chargement de la conversation archivée:', error);
        showError('Erreur de connexion au serveur');
    } finally {
        showLoadingIndicator(false);
    }
}

/**
 * Charge les détails complets de la conversation
 */
async function loadConversationDetails(conversation) {
    try {
        // Charger le contenu complet depuis l'archive
        const response = await fetch(`/portal/archives/${archiveDate}/${sessionId}`);
        
        if (response.ok) {
            const fullConversation = await response.json();
            displayConversationInfo(fullConversation);
            displayArchivedMessages(fullConversation.messages || []);
        } else {
            // Fallback sur les données de base si l'archive complète n'est pas accessible
            displayConversationInfo(conversation);
            displayArchivedMessages([]);
        }
        
        showChatInterface();
    } catch (error) {
        console.error('Erreur lors du chargement des détails:', error);
        // Fallback sur les données de base
        displayConversationInfo(conversation);
        displayArchivedMessages([]);
        showChatInterface();
    }
}

/**
 * Affiche les informations de la conversation
 */
function displayConversationInfo(conversation) {
    document.getElementById('sessionId').textContent = conversation.id;
    document.getElementById('createdAt').textContent = new Date(conversation.createdAt).toLocaleString('fr-FR');
    document.getElementById('terminatedAt').textContent = new Date(conversation.terminatedAt).toLocaleString('fr-FR');
    document.getElementById('terminatedBy').textContent = conversation.terminatedBy;
    document.getElementById('messageCount').textContent = conversation.messageCount;
}

/**
 * Affiche les messages archivés
 */
function displayArchivedMessages(messages) {
    const messagesList = document.getElementById('messagesList');
    
    if (messages.length === 0) {
        // Aucun message dans cette conversation
        messagesList.innerHTML = `
            <div class="welcome-message">
                <h3>💬 Aucun message</h3>
                <p>Cette conversation n'a aucun message enregistré.</p>
            </div>
        `;
        return;
    }

    // Afficher tous les messages
    messagesList.innerHTML = '';
    
    // Ajouter un message d'en-tête
    const headerMessage = document.createElement('div');
    headerMessage.className = 'welcome-message';
    headerMessage.innerHTML = `
        <h3>🗃️ Conversation archivée</h3>
        <p>${messages.length} message(s) au total</p>
    `;
    messagesList.appendChild(headerMessage);
    
    // Afficher chaque message
    messages.forEach(message => {
        const messageElement = createMessageElement(message);
        messagesList.appendChild(messageElement);
    });
    
    // Scroll vers le bas pour voir tous les messages
    setTimeout(() => {
        const messagesContainer = document.querySelector('.messages-container');
        if (messagesContainer) {
            messagesContainer.scrollTop = messagesContainer.scrollHeight;
        }
    }, 100);
}

/**
 * Crée un élément de message en lecture seule
 */
function createMessageElement(message) {
    const div = document.createElement('div');
    div.className = `message ${message.sender === 'technicien' ? 'sent' : 'received'} readonly`;
    
    const time = new Date(message.timestamp).toLocaleTimeString('fr-FR', {
        hour: '2-digit',
        minute: '2-digit'
    });
    
    if (message.type === 'text') {
        div.innerHTML = `
            <div class="message-content">${escapeHtml(message.content)}</div>
            <div class="message-time">${time}</div>
        `;
    } else if (message.type === 'image') {
        div.innerHTML = `
            <div class="message-content">
                <div class="image-container">
                    <img src="${message.content}" alt="Image partagée" class="message-image" onclick="openImageModal('${message.content}')">
                </div>
            </div>
            <div class="message-time">${time}</div>
        `;
    }
    
    return div;
}

/**
 * Affiche l'indicateur de chargement
 */
function showLoadingIndicator(show) {
    document.getElementById('loadingIndicator').style.display = show ? 'flex' : 'none';
}

/**
 * Affiche l'interface de chat
 */
function showChatInterface() {
    document.getElementById('loadingIndicator').style.display = 'none';
    document.getElementById('connectionError').style.display = 'none';
    document.getElementById('chatInterface').style.display = 'flex';
}

/**
 * Affiche une erreur
 */
function showError(message) {
    document.getElementById('loadingIndicator').style.display = 'none';
    document.getElementById('chatInterface').style.display = 'none';
    document.getElementById('errorMessage').textContent = message;
    document.getElementById('connectionError').style.display = 'flex';
}

/**
 * Ouvre une image en modal
 */
function openImageModal(imageSrc) {
    window.open(imageSrc, '_blank');
}

/**
 * Échappe les caractères HTML
 */
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
} 