/**
 * Chat JavaScript - Portail Vert
 * Gestion de la page de chat avec polling AJAX
 */

// Variables globales
let sessionId = null;
let lastMessageId = null;
let pollingInterval = null;
let isConnected = false;
let selectedImage = null;

// Initialisation au chargement de la page
document.addEventListener('DOMContentLoaded', function() {
    // Extraire l'ID de session depuis l'URL
    const urlParams = new URLSearchParams(window.location.search);
    sessionId = urlParams.get('session');
    
    if (!sessionId) {
        showError('Aucune session spécifiée dans l\'URL');
        return;
    }
    
    // Afficher l'ID de session
    document.getElementById('sessionId').textContent = `Session: ${sessionId}`;
    
    // Initialiser la connexion
    initializeChat();
});

/**
 * Initialise la session de chat
 */
async function initializeChat() {
    showLoadingIndicator(true);
    
    try {
        // Tenter de se connecter à la session
        const response = await fetch(`/api/chat/${sessionId}/connect`);
        const result = await response.json();
        
        if (response.ok && result.success) {
            isConnected = true;
            showChatInterface();
            setupEventListeners();
            loadInitialMessages();
            startPolling();
            updateStatus('🟢 Connecté');
        } else if (response.status === 410 && result.terminated) {
            // Conversation terminée
            showTerminatedConversation(result);
        } else {
            showError(result.error || 'Impossible de se connecter à la session');
        }
    } catch (error) {
        console.error('Erreur lors de la connexion:', error);
        showError('Erreur de connexion au serveur');
    } finally {
        showLoadingIndicator(false);
    }
}

/**
 * Configure tous les événements de la page de chat
 */
function setupEventListeners() {
    // Formulaire de message
    const messageForm = document.getElementById('messageForm');
    messageForm.addEventListener('submit', sendMessage);
    
    // Bouton d'image
    const imageBtn = document.getElementById('imageBtn');
    const imageInput = document.getElementById('imageInput');
    
    imageBtn.addEventListener('click', () => {
        imageInput.click();
    });
    
    imageInput.addEventListener('change', handleImageSelection);
    
    // Boutons de prévisualisation d'image
    const sendImageBtn = document.getElementById('sendImageBtn');
    const cancelImageBtn = document.getElementById('cancelImageBtn');
    
    sendImageBtn.addEventListener('click', sendImage);
    cancelImageBtn.addEventListener('click', cancelImage);
    
    // Bouton de retry de connexion
    const retryBtn = document.getElementById('retryConnection');
    retryBtn.addEventListener('click', initializeChat);
    
    // Fermeture de modal
    const modalOk = document.getElementById('modalOk');
    modalOk.addEventListener('click', hideModal);
    
    // Bouton pour descendre aux nouveaux messages
    const scrollToBottomBtn = document.getElementById('scrollToBottomBtn');
    scrollToBottomBtn.addEventListener('click', () => {
        scrollToBottom(true);
        hideNewMessagesIndicator();
    });
    
    // Raccourci clavier pour envoyer (Entrée)
    const messageInput = document.getElementById('messageInput');
    messageInput.addEventListener('keypress', (e) => {
        if (e.key === 'Enter' && !e.shiftKey) {
            e.preventDefault();
            sendMessage(e);
        }
    });
    
    // Gestion spéciale pour mobile : scroll automatique au focus
    messageInput.addEventListener('focus', () => {
        if (window.innerWidth <= 768) { // Mobile
            setTimeout(() => {
                forceScrollToBottom();
                // Aussi scroller la textbox en vue
                const inputContainer = document.getElementById('messageInput').closest('.input-container');
                if (inputContainer) {
                    inputContainer.scrollIntoView({ behavior: 'smooth', block: 'end' });
                }
            }, 300); // Délai pour attendre que le clavier apparaisse
        }
    });
    
    // Écouter le scroll pour détecter si l'utilisateur remonte
    const messagesContainer = document.querySelector('.messages-container');
    messagesContainer.addEventListener('scroll', () => {
        // Cacher l'indicateur si l'utilisateur descend manuellement
        if (isUserAtBottom()) {
            hideNewMessagesIndicator();
        }
    });
}

/**
 * Charge les messages initiaux
 */
async function loadInitialMessages() {
    try {
        const response = await fetch(`/api/chat/${sessionId}/messages`);
        const result = await response.json();
        
        if (response.ok) {
            displayMessages(result.messages);
            if (result.messages.length > 0) {
                lastMessageId = result.messages[result.messages.length - 1].id;
                // Forcer le scroll après le chargement initial
                setTimeout(() => {
                    forceScrollToBottom();
                }, 500);
            }
        }
    } catch (error) {
        console.error('Erreur lors du chargement des messages:', error);
    }
}

/**
 * Démarre le polling AJAX toutes les 3 secondes
 */
function startPolling() {
    if (pollingInterval) {
        clearInterval(pollingInterval);
    }
    
    pollingInterval = setInterval(async () => {
        if (!isConnected) return;
        
        try {
            const url = lastMessageId 
                ? `/api/chat/${sessionId}/messages?since=${lastMessageId}`
                : `/api/chat/${sessionId}/messages`;
                
            const response = await fetch(url);
            const result = await response.json();
            
            if (response.ok && result.messages.length > 0) {
                displayMessages(result.messages);
                lastMessageId = result.messages[result.messages.length - 1].id;
                
                // FORCER l'autoscroll pour les nouveaux messages du polling
                setTimeout(() => {
                    forceScrollToBottom();
                }, 100);
                
                // Mettre à jour le statut si des messages ont été reçus
                updateStatus('🟢 Connecté - Nouveaux messages');
                
                // Remettre le statut normal after 2 secondes
                setTimeout(() => {
                    updateStatus('🟢 Connecté');
                }, 2000);
            }
        } catch (error) {
            console.error('Erreur lors du polling:', error);
            updateStatus('🔴 Connexion interrompue');
        }
    }, 3000); // Polling toutes les 3 secondes
}

/**
 * Envoie un message texte
 */
async function sendMessage(e) {
    e.preventDefault();
    
    const messageInput = document.getElementById('messageInput');
    const message = messageInput.value.trim();
    
    if (!message) return;
    
    const sendBtn = document.getElementById('sendBtn');
    sendBtn.disabled = true;
    
    // Générer un ID temporaire unique
    const tempId = 'temp-' + Date.now();
    
    // Créer immédiatement le message dans l'interface (optimiste)
    const optimisticMessage = {
        id: tempId,
        content: message,
        sender: 'technicien',
        type: 'text',
        timestamp: new Date().toISOString(),
        status: 'pending' // État: pending, sent, failed
    };
    
    // Ajouter le message temporairement à l'interface
    const messageElement = createMessageElement(optimisticMessage);
    const messagesList = document.getElementById('messagesList');
    messagesList.appendChild(messageElement);
    
    // Vider la textbox immédiatement pour plus de fluidité
    messageInput.value = '';
    
    // FORCER immédiatement l'autoscroll
    setTimeout(() => {
        forceScrollToBottom();
    }, 10);
    
    // Timeout pour détecter les échecs (5 secondes)
    const failureTimeout = setTimeout(() => {
        markMessageAsFailed(tempId);
    }, 5000);
    
    try {
        const response = await fetch(`/api/chat/${sessionId}/message`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                message: message,
                sender: 'technicien'
            })
        });
        
        const result = await response.json();
        
        if (response.ok && result.success) {
            // Succès ! Annuler le timeout et marquer comme envoyé
            clearTimeout(failureTimeout);
            markMessageAsSent(tempId);
            
            console.log('✅ Message envoyé avec succès');
        } else if (response.status === 410 && result.terminated) {
            // Conversation terminée pendant l'envoi
            clearTimeout(failureTimeout);
            showTerminatedConversation(result);
            // Supprimer le message temporaire
            const tempMessage = document.getElementById(`message-${tempId}`);
            if (tempMessage) tempMessage.remove();
        } else {
            // Échec de l'envoi
            clearTimeout(failureTimeout);
            markMessageAsFailed(tempId, result.error || 'Erreur lors de l\'envoi du message');
        }
    } catch (error) {
        // Erreur de connexion
        clearTimeout(failureTimeout);
        console.error('Erreur lors de l\'envoi du message:', error);
        markMessageAsFailed(tempId, 'Erreur de connexion au serveur');
    } finally {
        sendBtn.disabled = false;
        messageInput.focus();
    }
}

/**
 * Gère la sélection d'une image
 */
function handleImageSelection(e) {
    const file = e.target.files[0];

    if (!file) return;

    // Vérifier le type de fichier
    if (!file.type.startsWith('image/')) {
        showModal('Erreur', 'Veuillez sélectionner un fichier image valide');
        return;
    }

    // Vérifier la taille (20MB max)
    if (file.size > 20 * 1024 * 1024) {
        showModal('Erreur', 'L\'image est trop volumineuse (maximum 20MB)');
        return;
    }

    selectedImage = file;

    // Afficher la prévisualisation
    const reader = new FileReader();
    reader.onload = (e) => {
        const previewImg = document.getElementById('previewImg');
        previewImg.src = e.target.result;
        const imagePreview = document.getElementById('imagePreview');
        imagePreview.style.display = 'block';

        // Faire défiler vers la prévisualisation pour qu'elle soit visible
        setTimeout(() => {
            imagePreview.scrollIntoView({
                behavior: 'smooth',
                block: 'nearest'
            });
        }, 100);
    };
    reader.readAsDataURL(file);
}

/**
 * Envoie l'image sélectionnée
 */
async function sendImage() {
    if (!selectedImage) return;

    const sendImageBtn = document.getElementById('sendImageBtn');
    const cancelImageBtn = document.getElementById('cancelImageBtn');
    const originalText = sendImageBtn.textContent;

    sendImageBtn.textContent = '📤 Envoi...';
    sendImageBtn.disabled = true;
    cancelImageBtn.disabled = true;
    
    // Générer un ID temporaire unique pour l'image
    const tempId = 'temp-img-' + Date.now();
    
    // Créer une URL temporaire pour l'image (optimiste)
    const tempImageUrl = URL.createObjectURL(selectedImage);
    
    // Créer immédiatement le message image dans l'interface
    const optimisticImageMessage = {
        id: tempId,
        content: tempImageUrl,
        sender: 'technicien',
        type: 'image',
        timestamp: new Date().toISOString(),
        status: 'pending'
    };
    
    // Ajouter l'image à l'interface immédiatement
    const messageElement = createMessageElement(optimisticImageMessage);
    const messagesList = document.getElementById('messagesList');
    messagesList.appendChild(messageElement);
    
    // Forcer l'autoscroll SANS annuler la sélection d'image encore
    setTimeout(() => {
        forceScrollToBottom();
    }, 10);
    
    // Timeout pour détecter les échecs (10 secondes pour les images)
    const failureTimeout = setTimeout(() => {
        markMessageAsFailed(tempId, 'Timeout - Image non envoyée');
    }, 10000);
    
    // Sauvegarder l'image avant de créer le FormData
    const imageToSend = selectedImage;
    
    console.log('🖼️ DEBUG Image:', {
        hasImage: !!imageToSend,
        imageSize: imageToSend ? imageToSend.size : 'N/A',
        imageName: imageToSend ? imageToSend.name : 'N/A',
        imageType: imageToSend ? imageToSend.type : 'N/A'
    });
    
    if (!imageToSend) {
        console.error('❌ Aucune image sélectionnée !');
        markMessageAsFailed(tempId, 'Aucune image sélectionnée');
        return;
    }
    
    try {
        const formData = new FormData();
        formData.append('image', imageToSend);
        formData.append('sender', 'technicien');
        
        console.log('📤 DEBUG FormData créé:', {
            hasImageEntry: formData.has('image'),
            hasSenderEntry: formData.has('sender')
        });
        
        const response = await fetch(`/api/chat/${sessionId}/upload`, {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (response.ok && result.success) {
            // Succès ! Annuler le timeout et marquer comme envoyé
            clearTimeout(failureTimeout);
            
            // Masquer la prévisualisation maintenant que c'est envoyé
            cancelImage();
            
            // Remplacer l'URL temporaire par l'URL réelle du serveur
            const realImageMessage = messageElement.querySelector('.message-image');
            if (realImageMessage && result.imageUrl) {
                realImageMessage.src = result.imageUrl;
                // Nettoyer l'URL temporaire
                URL.revokeObjectURL(tempImageUrl);
            }
            
            markMessageAsSent(tempId);
            console.log('✅ Image envoyée avec succès');
        } else if (response.status === 410 && result.terminated) {
            // Conversation terminée pendant l'envoi
            clearTimeout(failureTimeout);
            URL.revokeObjectURL(tempImageUrl);
            cancelImage();
            showTerminatedConversation(result);
            // Supprimer le message temporaire
            const tempMessage = document.getElementById(`message-${tempId}`);
            if (tempMessage) tempMessage.remove();
        } else {
            // Échec de l'envoi
            clearTimeout(failureTimeout);
            URL.revokeObjectURL(tempImageUrl);
            markMessageAsFailed(tempId, result.error || 'Erreur lors de l\'envoi de l\'image');
        }
    } catch (error) {
        // Erreur de connexion
        clearTimeout(failureTimeout);
        URL.revokeObjectURL(tempImageUrl);
        console.error('Erreur lors de l\'envoi de l\'image:', error);
        markMessageAsFailed(tempId, 'Erreur de connexion au serveur');
    } finally {
        sendImageBtn.textContent = originalText;
        sendImageBtn.disabled = false;
        cancelImageBtn.disabled = false;
    }
}

/**
 * Annule la sélection d'image
 */
function cancelImage() {
    selectedImage = null;
    document.getElementById('imageInput').value = '';
    document.getElementById('imagePreview').style.display = 'none';
}

/**
 * Affiche les messages dans l'interface
 */
function displayMessages(messages) {
    const messagesList = document.getElementById('messagesList');
    let newMessagesAdded = false;

    messages.forEach(message => {
        // Vérifier si le message existe déjà (y compris les messages temporaires)
        if (document.getElementById(`message-${message.id}`)) {
            return;
        }
        
        // Vérifier si c'est un message réel qui correspond à un message temporaire
        const tempMessages = messagesList.querySelectorAll('[id^="message-temp-"]');
        let replacedTempMessage = false;
        
        tempMessages.forEach(tempMsg => {
            // Pour les messages texte, comparer le contenu
            if (message.type === 'text') {
                const tempContent = tempMsg.querySelector('.message-content').textContent.trim();
                if (message.content === tempContent && message.sender === 'technicien') {
                    // Ne PAS supprimer le message temporaire, juste le marquer comme envoyé
                    const tempId = tempMsg.id.replace('message-', '');
                    markMessageAsSent(tempId);
                    
                    // Changer l'ID pour éviter les doublons futurs
                    tempMsg.id = `message-${message.id}`;
                    replacedTempMessage = true;
                }
            }
            // Pour les images, utiliser une logique plus robuste basée sur la proximité temporelle
            else if (message.type === 'image') {
                if (message.sender === 'technicien') {
                    // Vérifier si c'est un message temporaire d'image récent (dans les 2 dernières minutes)
                    const messageTimestamp = new Date(message.timestamp);
                    const tempMsgId = tempMsg.id;
                    
                    // Extraire le timestamp du message temporaire depuis son ID
                    if (tempMsgId.startsWith('message-temp-img-')) {
                        const tempTimestamp = parseInt(tempMsgId.replace('message-temp-img-', ''));
                        const tempDate = new Date(tempTimestamp);
                        
                        // Si les messages sont dans un délai de 2 minutes (120 secondes)
                        const timeDifference = Math.abs(messageTimestamp - tempDate) / 1000;
                        
                        if (timeDifference <= 120) { // 2 minutes de tolérance
                            console.log('🖼️ Image en double détectée (proximité temporelle), mise à jour du message temporaire');
                            
                            // Mettre à jour l'image avec l'URL réelle du serveur
                            const tempImg = tempMsg.querySelector('.message-image');
                            if (tempImg) {
                                // Libérer l'ancienne URL blob
                                if (tempImg.src.startsWith('blob:')) {
                                    URL.revokeObjectURL(tempImg.src);
                                }
                                tempImg.src = message.content;
                            }
                            
                            const tempId = tempMsg.id.replace('message-', '');
                            markMessageAsSent(tempId);
                            
                            // Changer l'ID pour éviter les doublons futurs
                            tempMsg.id = `message-${message.id}`;
                            replacedTempMessage = true;
                        }
                    }
                }
            }
        });

        // Ne créer un nouvel élément que si on n'a pas mis à jour un message temporaire
        if (!replacedTempMessage) {
            // Pour les messages du serveur (surtout ceux reçus), ajouter un statut "sent" par défaut
            if (message.sender === 'technicien' && !message.status) {
                message.status = 'sent';
            }
            
            const messageElement = createMessageElement(message);
            messagesList.appendChild(messageElement);
            newMessagesAdded = true;
        }
    });

    // TOUJOURS faire défiler vers le bas quand de nouveaux messages arrivent
    if (newMessagesAdded) {
        // Scroll immédiat
        forceScrollToBottom();
        
        // Scroll après le rendu des images (au cas où)
        setTimeout(() => {
            forceScrollToBottom();
        }, 300);
    }
}

/**
 * Crée un élément de message
 */
function createMessageElement(message) {
    const div = document.createElement('div');
    div.className = `message ${message.sender === 'technicien' ? 'sent' : 'received'}`;
    div.id = `message-${message.id}`;
    
    const time = new Date(message.timestamp).toLocaleTimeString('fr-FR', {
        hour: '2-digit',
        minute: '2-digit'
    });
    
    // Ajouter les classes d'état pour les messages optimistes
    if (message.status) {
        div.classList.add(`status-${message.status}`);
    }
    
    if (message.type === 'text') {
        div.innerHTML = `
            <div class="message-content">${escapeHtml(message.content)}</div>
            <div class="message-time">
                ${time}
                ${getMessageStatusIcon(message.status)}
            </div>
        `;
    } else if (message.type === 'image') {
        div.innerHTML = `
            <div class="message-content">
                <div class="image-container">
                    <img src="${message.content}" alt="Image partagée" class="message-image" onclick="openImageModal('${message.content}')" onload="handleImageLoad()">
                    <button class="download-btn" onclick="downloadImage('${message.content}')" title="Télécharger l'image">
                        📥
                    </button>
                </div>
            </div>
            <div class="message-time">
                ${time}
                ${getMessageStatusIcon(message.status)}
            </div>
        `;
    }
    
    return div;
}

/**
 * Retourne l'icône d'état d'un message
 */
function getMessageStatusIcon(status) {
    // console.log('🔍 Status icon pour:', status);
    
    switch (status) {
        case 'pending':
            return '<span class="status-icon pending" title="Envoi en cours...">⏳</span>';
        case 'sent':
            return '<span class="status-icon sent" title="Envoyé">✓</span>';
        case 'failed':
            return '<span class="status-icon failed" title="Échec - Cliquer pour réessayer">⚠️</span>';
        default:
            // Pour les messages sans statut (reçus), ne pas afficher d'icône
            // console.log('⚪ Aucun statut, pas d\'icône');
            return '';
    }
}

/**
 * Marque un message comme envoyé avec succès
 */
function markMessageAsSent(messageId) {
    // console.log('✅ Marquage comme envoyé pour:', messageId);
    
    const messageElement = document.getElementById(`message-${messageId}`);
    if (messageElement) {
        // console.log('📝 Élément trouvé, mise à jour du statut');
        
        messageElement.classList.remove('status-pending', 'status-failed');
        messageElement.classList.add('status-sent');
        
        // Mettre à jour l'icône d'état
        const statusIcon = messageElement.querySelector('.status-icon');
        if (statusIcon) {
            // console.log('🔄 Mise à jour de l\'icône de statut');
            statusIcon.className = 'status-icon sent';
            statusIcon.textContent = '✓';
            statusIcon.title = 'Envoyé';
        } else {
            // console.log('❌ Icône de statut non trouvée');
        }
    } else {
        // console.log('❌ Élément de message non trouvé pour ID:', messageId);
    }
}

/**
 * Marque un message comme échoué
 */
function markMessageAsFailed(messageId, errorMessage = 'Échec de l\'envoi') {
    const messageElement = document.getElementById(`message-${messageId}`);
    if (messageElement) {
        messageElement.classList.remove('status-pending', 'status-sent');
        messageElement.classList.add('status-failed');
        
        // Mettre à jour l'icône d'état
        const statusIcon = messageElement.querySelector('.status-icon');
        if (statusIcon) {
            statusIcon.className = 'status-icon failed';
            statusIcon.textContent = '⚠️';
            statusIcon.title = `${errorMessage} - Cliquer pour réessayer`;
            
            // Ajouter un gestionnaire de clic pour réessayer
            statusIcon.onclick = () => retryMessage(messageId);
            statusIcon.style.cursor = 'pointer';
        }
        
        console.error('❌ Message échec:', errorMessage);
    }
}

/**
 * Réessaie d'envoyer un message échoué
 */
function retryMessage(messageId) {
    const messageElement = document.getElementById(`message-${messageId}`);
    if (messageElement) {
        const messageContent = messageElement.querySelector('.message-content').textContent;
        
        // Supprimer le message échoué
        messageElement.remove();
        
        // Remettre le texte dans la textbox et essayer de renvoyer
        const messageInput = document.getElementById('messageInput');
        messageInput.value = messageContent;
        messageInput.focus();
        
        // Optionnel: envoyer automatiquement
        // setTimeout(() => {
        //     sendMessage(new Event('submit'));
        // }, 100);
    }
}

/**
 * Fait défiler la zone de messages vers le bas de façon FORCÉE
 */
function forceScrollToBottom() {
    const messagesContainer = document.querySelector('.messages-container');
    if (!messagesContainer) {
        // console.log('❌ DEBUG: messagesContainer non trouvé');
        return;
    }
    
    // Debug complet des dimensions (gardé pour diagnostics)
    const containerHeight = messagesContainer.clientHeight;
    const containerScrollHeight = messagesContainer.scrollHeight;
    
    // Debug désactivé en production
    // console.log(`🔍 DEBUG: clientHeight: ${containerHeight}, scrollHeight: ${containerScrollHeight}`);
    
    // Si le conteneur n'a pas de hauteur définie, on ne peut pas scroller
    if (containerHeight === 0 || containerScrollHeight <= containerHeight) {
        // console.log('⚠️ DEBUG: Conteneur trop petit pour scroller');
        return;
    }
    
    // Détecter si on est sur mobile pour un scroll instantané
    const isMobile = window.innerWidth <= 768;
    
    // Sur mobile, ajouter du padding supplémentaire pour compenser le clavier
    const extraScrollMobile = isMobile ? 150 : 0; // Augmenté de 100 à 150px
    
    // Une seule tentative de scroll, propre et sans sautillement
    messagesContainer.scrollTo({
        top: containerScrollHeight + extraScrollMobile,
        behavior: isMobile ? 'instant' : 'smooth' // Instantané sur mobile, smooth sur PC
    });
    
    // Sur mobile, faire un scroll supplémentaire après un délai pour le clavier
    if (isMobile) {
        setTimeout(() => {
            messagesContainer.scrollTo({
                top: containerScrollHeight + extraScrollMobile,
                behavior: 'instant'
            });
        }, 300); // Délai pour que le clavier se stabilise
    }
    
    // console.log(`✅ DEBUG: Scroll terminé (${isMobile ? 'mobile' : 'desktop'}): ${messagesContainer.scrollTop}`);
}

/**
 * Fait défiler la zone de messages vers le bas (version simplifiée)
 */
function scrollToBottom(force = false) {
    if (force) {
        forceScrollToBottom();
        return;
    }
    
    const messagesContainer = document.querySelector('.messages-container');
    if (!messagesContainer) return;
    
    // Toujours faire défiler vers le bas pour les nouveaux messages
    messagesContainer.scrollTop = messagesContainer.scrollHeight;
}

/**
 * Vérifie si l'utilisateur est en bas de la conversation
 */
function isUserAtBottom() {
    const messagesContainer = document.querySelector('.messages-container');
    if (!messagesContainer) return true;
    
    return messagesContainer.scrollTop + messagesContainer.clientHeight >=
           messagesContainer.scrollHeight - 50; // Marge réduite pour être plus sensible
}

/**
 * Gère le chargement d'une image (pour déclencher l'autoscroll)
 */
function handleImageLoad() {
    // Quand une image se charge, TOUJOURS forcer le scroll
    setTimeout(() => {
        forceScrollToBottom();
    }, 50);
}

/**
 * Télécharge une image
 */
function downloadImage(imageSrc) {
    const link = document.createElement('a');
    link.href = imageSrc;
    link.download = `image_${Date.now()}.jpg`; // Nom de fichier avec timestamp
    link.target = '_blank';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

/**
 * Affiche l'indicateur de nouveaux messages
 */
function showNewMessagesIndicator() {
    const indicator = document.getElementById('newMessagesIndicator');
    if (indicator) {
        indicator.style.display = 'block';
    }
}

/**
 * Cache l'indicateur de nouveaux messages
 */
function hideNewMessagesIndicator() {
    const indicator = document.getElementById('newMessagesIndicator');
    if (indicator) {
        indicator.style.display = 'none';
    }
}

/**
 * Affiche l'interface de chat
 */
function showChatInterface() {
    document.getElementById('loadingIndicator').style.display = 'none';
    document.getElementById('connectionError').style.display = 'none';
    document.getElementById('chatInterface').style.display = 'flex';
}

/**
 * Affiche une erreur de connexion
 */
function showError(message) {
    document.getElementById('loadingIndicator').style.display = 'none';
    document.getElementById('chatInterface').style.display = 'none';
    document.getElementById('errorMessage').textContent = message;
    document.getElementById('connectionError').style.display = 'flex';
}

/**
 * Affiche/cache l'indicateur de chargement
 */
function showLoadingIndicator(show) {
    document.getElementById('loadingIndicator').style.display = show ? 'flex' : 'none';
}

/**
 * Met à jour le statut de connexion
 */
function updateStatus(status) {
    document.getElementById('sessionStatus').textContent = status;
}

/**
 * Affiche une modal d'information
 */
function showModal(title, message) {
    document.getElementById('modalTitle').textContent = title;
    document.getElementById('modalMessage').textContent = message;
    document.getElementById('infoModal').style.display = 'flex';
}

/**
 * Cache la modal
 */
function hideModal() {
    document.getElementById('infoModal').style.display = 'none';
}

/**
 * Ouvre une image en modal (fonction future)
 */
function openImageModal(imageSrc) {
    // Pour l'instant, ouvrir dans un nouvel onglet
    window.open(imageSrc, '_blank');
}

/**
 * Échappe les caractères HTML pour éviter les injections XSS
 */
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

/**
 * Affiche l'interface pour conversation terminée
 */
function showTerminatedConversation(result) {
    // Arrêter le polling
    if (pollingInterval) {
        clearInterval(pollingInterval);
        pollingInterval = null;
    }
    
    isConnected = false;
    
    // Charger les derniers messages pour que l'utilisateur puisse les voir
    loadInitialMessages();
    
    // Désactiver l'interface d'envoi
    const messageForm = document.getElementById('messageForm');
    const messageInput = document.getElementById('messageInput');
    const sendBtn = document.getElementById('sendBtn');
    const imageBtn = document.getElementById('imageBtn');
    const inputContainer = document.querySelector('.input-container');
    
    if (messageForm) messageForm.style.display = 'none';
    
    // Créer un message d'information sur la terminaison
    const terminationDate = new Date(result.terminatedAt).toLocaleString('fr-FR');
    const terminationMessage = `
        <div style="
            background-color: #fff3cd;
            border: 1px solid #ffeaa7;
            border-radius: 8px;
            padding: 15px;
            margin: 10px;
            text-align: center;
            color: #856404;
        ">
            <h3 style="margin: 0 0 10px 0; color: #856404;">
                🛑 Conversation terminée
            </h3>
            <p style="margin: 5px 0;">
                Cette conversation a été terminée par <strong>${result.terminatedBy}</strong>
            </p>
            <p style="margin: 5px 0; font-size: 14px;">
                Le ${terminationDate}
            </p>
            <p style="margin: 10px 0 0 0; font-weight: bold;">
                Vous ne pouvez plus envoyer de messages.
            </p>
        </div>
    `;
    
    // Insérer le message après la zone de messages
    const chatInterface = document.getElementById('chatInterface');
    if (chatInterface) {
        chatInterface.insertAdjacentHTML('beforeend', terminationMessage);
    }
    
    // Mettre à jour le statut
    updateStatus('🛑 Conversation terminée');
    
    // Scroll vers le bas pour voir le message de terminaison
    setTimeout(() => {
        forceScrollToBottom();
    }, 100);
    
    console.log('💔 Conversation terminée:', result);
}

// Nettoyage lors de la fermeture de la page
window.addEventListener('beforeunload', () => {
    if (pollingInterval) {
        clearInterval(pollingInterval);
    }
});
