/**
 * Génère un lien de chat (production ou local)
 */
function generateChatLink(sessionId) {
    const hostname = window.location.hostname;
    const protocol = window.location.protocol;
    
    // En production sur mathix.xyz
    if (hostname === 'mathix.xyz' || hostname.includes('mathix.xyz')) {
        return `https://mathix.xyz/chat.html?session=${sessionId}`;
    }
    
    // Pour les tests locaux
    const isLocal = hostname === 'localhost' ||
                   hostname.startsWith('192.168.') ||
                   hostname.startsWith('10.') ||
                   hostname.startsWith('172.');
    
    if (isLocal && window.location.port) {
        return `${protocol}//${hostname}:${window.location.port}/chat.html?session=${sessionId}`;
    }
    
    return `${protocol}//${hostname}/chat.html?session=${sessionId}`;
}

/**
 * Copie le lien d'une conversation
 */
async function copyConversationLink(sessionId) {
    const chatLink = generateChatLink(sessionId);

    try {
        await navigator.clipboard.writeText(chatLink);
        showNotification('Lien de conversation copié !', 'success');
    } catch (error) {
        console.error('Erreur lors de la copie:', error);
        showNotification('Erreur lors de la copie du lien', 'error');
    }
}

/**
 * Affiche la modal de déconnexion
 */
function showLogoutModal() {
    document.getElementById('logoutModal').style.display = 'flex';
}

/**
 * Cache la modal de déconnexion
 */
function hideLogoutModal() {
    document.getElementById('logoutModal').style.display = 'none';
}

/**
 * Déconnecte l'utilisateur
 */
async function logout() {
    try {
        await fetch('/api/logout', { method: 'POST' });
    } catch (error) {
        console.error('Erreur lors de la déconnexion:', error);
    } finally {
        sessionStorage.removeItem('authenticated');
        window.location.href = '/login.html';
    }
}

/**
 * Affiche une notification
 */
function showNotification(message, type = 'info') {
    // Créer l'élément de notification
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.textContent = message;
    
    // Calculer la position en fonction des notifications existantes
    const existingNotifications = document.querySelectorAll('.notification');
    const topOffset = 20 + (existingNotifications.length * 110); // 110px d'espacement entre les notifications
    
    // Styles de base pour la notification
    Object.assign(notification.style, {
        position: 'fixed',
        top: `${topOffset}px`,
        right: '20px',
        padding: '15px 20px',
        borderRadius: '8px',
        color: 'white',
        fontWeight: '500',
        zIndex: '9999',
        maxWidth: '380px',
        minHeight: '60px',
        boxShadow: '0 4px 20px rgba(0, 0, 0, 0.2)',
        transform: 'translateX(100%)',
        transition: 'transform 0.3s ease-out, top 0.3s ease-out',
        wordWrap: 'break-word',
        fontSize: '14px',
        lineHeight: '1.5',
        marginBottom: '10px'
    });
    
    // Couleurs selon le type
    const colors = {
        success: '#28a745',
        error: '#dc3545',
        warning: '#ffc107',
        info: '#17a2b8'
    };
    
    notification.style.backgroundColor = colors[type] || colors.info;
    
    // Ajouter au DOM
    document.body.appendChild(notification);
    
    // Animation d'entrée
    setTimeout(() => {
        notification.style.transform = 'translateX(0)';
    }, 100);
    
    // Fonction pour repositionner les notifications restantes
    const repositionNotifications = () => {
        const currentNotifications = document.querySelectorAll('.notification');
        currentNotifications.forEach((notif, index) => {
            notif.style.top = `${20 + (index * 110)}px`;
        });
    };
    
    // Suppression automatique après 7 secondes (un peu plus long pour lire)
    setTimeout(() => {
        notification.style.transform = 'translateX(100%)';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
                // Repositionner les notifications restantes
                repositionNotifications();
            }
        }, 300);
    }, 7000);
}

/**
 * Vérifie si l'utilisateur est administrateur
 */
async function checkAdminStatus() {
    try {
        const response = await fetch('/admin/managers');
        if (response.ok) {
            // Si la requête réussit, l'utilisateur est admin
            document.getElementById('adminBtn').style.display = 'inline-block';
        }
    } catch (error) {
        // Ignorer les erreurs, le bouton reste caché
    }
}

/**
 * Gère les actions des conversations (view, copy, terminate)
 */
function handleConversationAction(event) {
    const button = event.target.closest('button[data-action]');
    
    // Ne traiter que les boutons avec data-action ET dans une conversation
    if (!button || !button.closest('.conversation-item')) return;

    const action = button.dataset.action;
    const sessionId = button.getAttribute('data-session-id');

    if (!sessionId || sessionId === 'null' || sessionId === 'undefined') {
        console.error('❌ sessionId non valide dans le bouton:', sessionId);
        showNotification('Erreur: ID de session manquant sur le bouton', 'error');
        return;
    }

    switch (action) {
        case 'view':
            openChatWindow(sessionId);
            break;
        case 'copy':
            copyConversationLink(sessionId);
            break;
        case 'terminate':
            terminateChat(sessionId);
            break;
    }
}

/**
 * Termine une session de chat - Affiche la modal de vérification des données sensibles
 */
function terminateChat(sessionId) {
    if (!sessionId) {
        console.error('❌ sessionId est vide ou undefined dans terminateChat');
        showNotification('Erreur: ID de session manquant', 'error');
        return;
    }
    
    sessionToTerminate = sessionId;
    showTerminateModal();
}

/**
 * Affiche la modal de terminaison avec vérification des données sensibles
 */
function showTerminateModal() {
    document.getElementById('terminateModal').style.display = 'flex';
}

/**
 * Cache la modal de terminaison
 */
function hideTerminateModal() {
    document.getElementById('terminateModal').style.display = 'none';
    // Ne pas remettre sessionToTerminate à null ici car elle est utilisée après
}

// Variable pour éviter les doubles clics sur la terminaison
let terminationInProgress = false;

/**
 * Gère la confirmation de terminaison selon la présence de données sensibles
 */
async function handleTerminateConfirmation(hasSensitiveData) {
    if (!sessionToTerminate) {
        console.error('❌ sessionToTerminate est null/undefined');
        showNotification('Erreur: Aucune session sélectionnée pour terminaison', 'error');
        return;
    }

    // Éviter les doubles clics
    if (terminationInProgress) {
        console.log('⚠️ Terminaison déjà en cours, ignorer le clic');
        return;
    }

    terminationInProgress = true;
    hideTerminateModal();

    try {
        const response = await fetch(`/portal/conversations/${sessionToTerminate}`, {
            method: 'DELETE',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ hasSensitiveData: hasSensitiveData })
        });

        const result = await response.json();

        if (response.ok) {
            if (result.success) {
                // Succès - archivage ou suppression réussie
                if (result.action === 'archived') {
                    showNotification('✅ Session terminée et archivée avec succès', 'success');
                } else if (result.action === 'deleted') {
                    showNotification('🗑️ Session supprimée complètement en raison de données sensibles', 'success');
                } else if (result.action === 'processing') {
                    // Traitement déjà en cours - pas besoin d'afficher d'erreur
                    console.log('⚠️ Traitement déjà en cours pour cette session');
                    // Nettoyer les variables avant de sortir
                    sessionToTerminate = null;
                    terminationInProgress = false;
                    return; // Sortir sans actualiser ni afficher de notification
                }
            } else {
                // Échec avec détection automatique de données sensibles
                if (result.action === 'deleted_auto') {
                    showNotification('⚠️ ' + result.error, 'warning');
                } else {
                    showNotification(result.error || 'Erreur lors de la terminaison de la session', 'error');
                }
            }

            // Actualiser la liste des conversations et les archives
            setTimeout(() => {
                loadConversations(true);
                loadArchives(false);
            }, 1000);
        } else {
            // Gestion des erreurs HTTP spécifiques
            if (response.status === 404) {
                showNotification('❌ Session non trouvée - elle a peut-être déjà été supprimée', 'error');
            } else if (response.status === 400) {
                showNotification('⚠️ ' + (result.error || 'Cette session a déjà été terminée'), 'warning');
            } else {
                showNotification(result.error || 'Erreur lors de la terminaison de la session', 'error');
            }
            
            // Actualiser la liste même en cas d'erreur pour refléter l'état actuel
            setTimeout(() => {
                loadConversations(true);
            }, 500);
        }
    } catch (error) {
        console.error('Erreur lors de la terminaison de la session:', error);
        showNotification('Erreur de connexion au serveur - Vérifiez votre connexion', 'error');
        // Nettoyer en cas d'erreur aussi
        terminationInProgress = false;
    }

    // Nettoyer les variables
    sessionToTerminate = null;
    terminationInProgress = false;
}

/**
 * Charge la liste des archives
 */
async function loadArchives(showLoading = false) {
    const loadingDiv = document.getElementById('archivesLoading');
    const listDiv = document.getElementById('archivesList');
    const emptyDiv = document.getElementById('noArchives');
    
    if (showLoading) {
        loadingDiv.style.display = 'block';
        emptyDiv.style.display = 'none';
    }
    
    try {
        const response = await fetch('/portal/archives');
        const archives = await response.json();
        
        if (response.ok) {
            displayArchives(archives);
        } else {
            console.error('Erreur lors du chargement des archives:', archives.error);
            showNotification('Erreur lors du chargement des archives', 'error');
        }
    } catch (error) {
        console.error('Erreur lors du chargement des archives:', error);
        showNotification('Erreur de connexion au serveur', 'error');
    } finally {
        loadingDiv.style.display = 'none';
    }
}

/**
 * Affiche la liste des dates d'archives
 */
function displayArchives(archives) {
    const listDiv = document.getElementById('archivesList');
    const emptyDiv = document.getElementById('noArchives');

    if (archives.length === 0) {
        emptyDiv.style.display = 'block';
        listDiv.innerHTML = '';
        return;
    }

    emptyDiv.style.display = 'none';
    listDiv.innerHTML = '';

    archives.forEach(archive => {
        const dateElement = document.createElement('div');
        dateElement.className = 'conversation-item';
        dateElement.innerHTML = `
            <div class="conversation-header">
                <div class="conversation-id">📅 ${new Date(archive.date).toLocaleDateString('fr-FR', {
                    weekday: 'long',
                    year: 'numeric',
                    month: 'long',
                    day: 'numeric'
                })}</div>
                <div class="conversation-status status-connected">${archive.conversationCount} conversation(s)</div>
            </div>
            <div class="conversation-actions">
                <button class="btn-primary btn-small" onclick="viewDailyArchives('${archive.date}')">
                    👁️ Voir les conversations
                </button>
            </div>
        `;
        listDiv.appendChild(dateElement);
    });
}

/**
 * Affiche les conversations archivées d'une date spécifique
 */
async function viewDailyArchives(date) {
    const loadingDiv = document.getElementById('archivesLoading');
    const archivesList = document.getElementById('archivesList');
    const dailyArchives = document.getElementById('dailyArchives');
    const dailyArchivesList = document.getElementById('dailyArchivesList');
    const dailyArchivesTitle = document.getElementById('dailyArchivesTitle');
    
    loadingDiv.style.display = 'block';
    
    try {
        const response = await fetch(`/portal/archives?date=${date}`);
        const conversations = await response.json();
        
        if (response.ok) {
            // Masquer la liste des dates et afficher les conversations du jour
            archivesList.style.display = 'none';
            dailyArchives.style.display = 'block';
            
            // Mettre à jour le titre
            const dateFormatted = new Date(date).toLocaleDateString('fr-FR', {
                weekday: 'long',
                year: 'numeric',
                month: 'long',
                day: 'numeric'
            });
            dailyArchivesTitle.textContent = `📁 Conversations du ${dateFormatted}`;
            
            // Afficher les conversations
            displayArchivedConversations(conversations);
        } else {
            showNotification('Erreur lors du chargement des conversations archivées', 'error');
        }
    } catch (error) {
        console.error('Erreur lors du chargement des conversations archivées:', error);
        showNotification('Erreur de connexion au serveur', 'error');
    } finally {
        loadingDiv.style.display = 'none';
    }
}

/**
 * Affiche les conversations archivées
 */
function displayArchivedConversations(conversations) {
    const listDiv = document.getElementById('dailyArchivesList');
    listDiv.innerHTML = '';

    conversations.forEach(conversation => {
        const element = document.createElement('div');
        element.className = 'conversation-item';
        element.dataset.conversationId = conversation.id;

        const createdDate = new Date(conversation.createdAt).toLocaleString('fr-FR');
        const terminatedDate = new Date(conversation.terminatedAt).toLocaleString('fr-FR');

        element.innerHTML = `
            <div class="conversation-header">
                <div class="conversation-id">Session: ${conversation.id}</div>
                <div class="conversation-status" style="background-color: #6c757d; color: white;">🗃️ Archivée</div>
            </div>
            <div class="conversation-meta">
                <span>📅 Créée: ${createdDate}</span>
                <span>💬 Messages: ${conversation.messageCount}</span>
                <span>🛑 Terminée: ${terminatedDate}</span>
                <span>👤 Par: ${conversation.terminatedBy}</span>
            </div>
            <div class="conversation-actions">
                <button class="btn-secondary btn-small" onclick="openArchivedChatWindow('${conversation.id}', '${conversation.date}')">
                    👁️ Voir la conversation
                </button>
            </div>
        `;
        
        listDiv.appendChild(element);
    });
}

/**
 * Retourne à la liste des dates d'archives
 */
function backToArchivesList() {
    document.getElementById('archivesList').style.display = 'block';
    document.getElementById('dailyArchives').style.display = 'none';
}

/**
 * Ouvre une conversation archivée en lecture seule
 */
function openArchivedChatWindow(sessionId, date) {
    const archiveUrl = `/archive-viewer.html?session=${sessionId}&date=${date}`;
    window.open(archiveUrl, '_blank', 'width=1000,height=700,scrollbars=yes,resizable=yes');
}

// Initialiser le tableau de bord au chargement de la page
document.addEventListener('DOMContentLoaded', initializeDashboard);
