/**
 * Login 2FA JavaScript - Portail Vert
 * Gestion de la vérification TOTP lors de la connexion
 */

// Vérification au chargement de la page
document.addEventListener('DOMContentLoaded', function() {
    // Vérifier si l'utilisateur a passé l'étape de mot de passe
    if (sessionStorage.getItem('password_verified') !== 'true') {
        window.location.href = '/login.html';
        return;
    }
    
    // Vérifier si déjà complètement authentifié
    if (sessionStorage.getItem('authenticated') === 'true') {
        window.location.href = '/dashboard.html';
        return;
    }
    
    // Initialiser la page de vérification 2FA
    initialize2FALogin();
});

/**
 * Initialise la page de connexion 2FA
 */
function initialize2FALogin() {
    setupEventListeners();
    focusOnTOTPField();
}

/**
 * Configure tous les événements
 */
function setupEventListeners() {
    // Formulaire de vérification TOTP
    document.getElementById('totpLoginForm').addEventListener('submit', handleTOTPLogin);
    
    // Bouton retour à la connexion
    document.getElementById('backToLogin').addEventListener('click', backToLogin);
    
    // Formatage automatique du code TOTP
    document.getElementById('totpCode').addEventListener('input', formatTOTPCode);
    
    // Auto-submit quand 6 chiffres sont saisis
    document.getElementById('totpCode').addEventListener('input', checkAutoSubmit);
    
    // Gérer l'entrée au clavier
    document.getElementById('totpCode').addEventListener('keydown', handleKeyDown);
}

/**
 * Gère la vérification TOTP
 */
async function handleTOTPLogin(e) {
    e.preventDefault();
    
    const totpBtn = document.getElementById('totpLoginBtn');
    const btnText = totpBtn.querySelector('.btn-text');
    const btnLoading = totpBtn.querySelector('.btn-loading');
    const errorDiv = document.getElementById('totpError');
    const totpCode = document.getElementById('totpCode').value;
    
    // Validation côté client
    if (totpCode.length !== 6 || !/^\d{6}$/.test(totpCode)) {
        showError(errorDiv, 'Le code doit contenir exactement 6 chiffres');
        return;
    }
    
    // Afficher l'état de chargement
    showLoading(totpBtn, btnText, btnLoading);
    hideError(errorDiv);
    
    try {
        const response = await fetch('/api/verify-totp-login', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ token: totpCode })
        });
        
        const result = await response.json();
        
        if (response.ok && result.success) {
            // Authentification complète réussie
            sessionStorage.setItem('authenticated', 'true');
            sessionStorage.removeItem('password_verified');
            
            // Redirection vers le dashboard
            window.location.href = result.redirect || '/dashboard.html';
        } else {
            showError(errorDiv, result.error || 'Code de vérification incorrect');
            
            // Vider le champ et le refocaliser
            document.getElementById('totpCode').value = '';
            focusOnTOTPField();
            
            // Si l'erreur indique une session expirée, retourner à la connexion
            if (result.error && result.error.includes('session')) {
                setTimeout(() => {
                    backToLogin();
                }, 2000);
            }
        }
    } catch (error) {
        console.error('Erreur lors de la vérification TOTP:', error);
        showError(errorDiv, 'Erreur de connexion au serveur');
        
        // Vider le champ et le refocaliser
        document.getElementById('totpCode').value = '';
        focusOnTOTPField();
    } finally {
        hideLoading(totpBtn, btnText, btnLoading);
    }
}

/**
 * Retour à la page de connexion
 */
function backToLogin() {
    // Nettoyer les données de session
    sessionStorage.removeItem('password_verified');
    sessionStorage.removeItem('authenticated');
    
    // Rediriger vers la connexion
    window.location.href = '/login.html';
}

/**
 * Formate le code TOTP (6 chiffres uniquement)
 */
function formatTOTPCode(e) {
    let value = e.target.value.replace(/\D/g, ''); // Supprimer tous les non-chiffres
    if (value.length > 6) {
        value = value.substring(0, 6);
    }
    e.target.value = value;
}

/**
 * Vérifie si auto-submit est possible
 */
function checkAutoSubmit(e) {
    const totpCode = e.target.value;
    if (totpCode.length === 6 && /^\d{6}$/.test(totpCode)) {
        // Attendre un petit délai pour que l'utilisateur puisse voir le code complet
        setTimeout(() => {
            document.getElementById('totpLoginForm').dispatchEvent(new Event('submit'));
        }, 500);
    }
}

/**
 * Gère les touches spéciales
 */
function handleKeyDown(e) {
    // Autoriser les touches de navigation et de modification
    const allowedKeys = [
        'Backspace', 'Delete', 'Tab', 'Escape', 'Enter',
        'ArrowLeft', 'ArrowRight', 'ArrowUp', 'ArrowDown',
        'Home', 'End'
    ];
    
    // Autoriser Ctrl+A, Ctrl+C, Ctrl+V, Ctrl+X
    if (e.ctrlKey && ['a', 'c', 'v', 'x'].includes(e.key.toLowerCase())) {
        return;
    }
    
    // Autoriser les touches autorisées et les chiffres
    if (allowedKeys.includes(e.key) || (e.key >= '0' && e.key <= '9')) {
        return;
    }
    
    // Bloquer toutes les autres touches
    e.preventDefault();
}

/**
 * Met le focus sur le champ TOTP
 */
function focusOnTOTPField() {
    setTimeout(() => {
        const totpField = document.getElementById('totpCode');
        totpField.focus();
        totpField.select();
    }, 100);
}

/**
 * Affiche un état de chargement
 */
function showLoading(button, textSpan, loadingSpan) {
    textSpan.style.display = 'none';
    loadingSpan.style.display = 'inline';
    button.disabled = true;
}

/**
 * Cache l'état de chargement
 */
function hideLoading(button, textSpan, loadingSpan) {
    textSpan.style.display = 'inline';
    loadingSpan.style.display = 'none';
    button.disabled = false;
}

/**
 * Affiche un message d'erreur
 */
function showError(errorDiv, message) {
    errorDiv.textContent = message;
    errorDiv.style.display = 'block';
    
    // Animation d'erreur
    errorDiv.style.animation = 'shake 0.5s ease-in-out';
    setTimeout(() => {
        errorDiv.style.animation = '';
    }, 500);
}

/**
 * Cache un message d'erreur
 */
function hideError(errorDiv) {
    errorDiv.style.display = 'none';
} 