/**
 * Setup 2FA JavaScript - Portail Vert
 * Gestion de la configuration de l'authentification à deux facteurs
 */

let currentStep = 1;
let secretKey = '';

// Vérification de l'authentification au chargement de la page
document.addEventListener('DOMContentLoaded', function() {
    // Vérifier si l'utilisateur est déjà connecté avec 2FA
    if (sessionStorage.getItem('totp_setup_complete') === 'true') {
        window.location.href = '/dashboard.html';
        return;
    }
    
    // Vérifier si l'utilisateur doit configurer le 2FA
    if (sessionStorage.getItem('requires_2fa_setup') !== 'true') {
        window.location.href = '/login.html';
        return;
    }
    
    // Initialiser la configuration 2FA
    initialize2FASetup();
});

/**
 * Initialise la configuration 2FA
 */
function initialize2FASetup() {
    setupEventListeners();
    showStep(1);
}

/**
 * Configure tous les événements
 */
function setupEventListeners() {
    // Formulaire de changement de mot de passe
    document.getElementById('passwordForm').addEventListener('submit', handlePasswordChange);
    
    // Formulaire de vérification TOTP
    document.getElementById('totpForm').addEventListener('submit', handleTOTPVerification);
    
    // Bouton de copie de la clé secrète
    document.getElementById('copySecret').addEventListener('click', copySecretKey);
    
    // Bouton "Aller au tableau de bord"
    document.getElementById('goToDashboard').addEventListener('click', () => {
        window.location.href = '/dashboard.html';
    });
    
    // Validation en temps réel du formulaire de mot de passe
    document.getElementById('newPassword').addEventListener('input', showPasswordStrength);
    document.getElementById('confirmPassword').addEventListener('input', validatePasswordMatch);
    
    // Formatage automatique du code TOTP
    document.getElementById('totpCode').addEventListener('input', formatTOTPCode);
}

/**
 * Affiche une étape spécifique
 */
function showStep(step) {
    // Cacher toutes les étapes
    document.querySelectorAll('.setup-step').forEach(stepEl => {
        stepEl.style.display = 'none';
    });
    
    // Afficher l'étape demandée
    switch(step) {
        case 1:
            document.getElementById('passwordStep').style.display = 'block';
            break;
        case 2:
            document.getElementById('totpStep').style.display = 'block';
            initializeTOTPSetup();
            break;
        case 3:
            document.getElementById('successStep').style.display = 'block';
            // Redirection automatique après 3 secondes
            setTimeout(() => {
                window.location.href = '/dashboard.html';
            }, 3000);
            break;
    }
    
    currentStep = step;
}

/**
 * Gère le changement de mot de passe
 */
async function handlePasswordChange(e) {
    e.preventDefault();
    
    const passwordBtn = document.getElementById('passwordBtn');
    const btnText = passwordBtn.querySelector('.btn-text');
    const btnLoading = passwordBtn.querySelector('.btn-loading');
    const errorDiv = document.getElementById('passwordError');
    
    // Validation côté client
    const newPassword = document.getElementById('newPassword').value;
    const confirmPassword = document.getElementById('confirmPassword').value;
    
    if (newPassword !== confirmPassword) {
        showError(errorDiv, 'Les mots de passe ne correspondent pas');
        return;
    }
    
    // Validation robuste du mot de passe
    const passwordValidation = validatePasswordStrength(newPassword);
    if (!passwordValidation.isValid) {
        showError(errorDiv, passwordValidation.errors.join('<br>'));
        return;
    }
    
    // Afficher l'état de chargement
    showLoading(passwordBtn, btnText, btnLoading);
    hideError(errorDiv);
    
    const formData = new FormData(e.target);
    const passwordData = {
        currentPassword: formData.get('currentPassword'),
        newPassword: formData.get('newPassword')
    };
    
    try {
        const response = await fetch('/api/change-password-2fa', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(passwordData)
        });
        
        const result = await response.json();
        
        if (response.ok && result.success) {
            showStep(2);
        } else {
            let errorMessage = result.error || 'Erreur lors du changement de mot de passe';
            if (result.details && Array.isArray(result.details)) {
                errorMessage += '<br><br>' + result.details.map(detail => `• ${detail}`).join('<br>');
            }
            showError(errorDiv, errorMessage);
        }
    } catch (error) {
        console.error('Erreur lors du changement de mot de passe:', error);
        showError(errorDiv, 'Erreur de connexion au serveur');
    } finally {
        hideLoading(passwordBtn, btnText, btnLoading);
    }
}

/**
 * Initialise la configuration TOTP (génération QR code)
 */
async function initializeTOTPSetup() {
    try {
        const response = await fetch('/api/generate-totp-secret', {
            method: 'POST'
        });
        
        const result = await response.json();
        
        if (response.ok && result.success) {
            secretKey = result.secret;
            document.getElementById('secretKey').value = result.secret;
            
            // Générer le QR code
            generateQRCode(result.qrCodeUrl);
        } else {
            showError(document.getElementById('totpError'), result.error || 'Erreur lors de la génération du secret TOTP');
        }
    } catch (error) {
        console.error('Erreur lors de l\'initialisation TOTP:', error);
        showError(document.getElementById('totpError'), 'Erreur de connexion au serveur');
    }
}

/**
 * Génère et affiche le QR code
 */
function generateQRCode(qrCodeUrl) {
    const qrContainer = document.getElementById('qrcode-container');
    qrContainer.innerHTML = `<img src="${qrCodeUrl}" alt="QR Code pour configuration 2FA" style="max-width: 256px; height: auto;">`;
}

/**
 * Gère la vérification TOTP
 */
async function handleTOTPVerification(e) {
    e.preventDefault();
    
    const totpBtn = document.getElementById('totpBtn');
    const btnText = totpBtn.querySelector('.btn-text');
    const btnLoading = totpBtn.querySelector('.btn-loading');
    const errorDiv = document.getElementById('totpError');
    
    // Afficher l'état de chargement
    showLoading(totpBtn, btnText, btnLoading);
    hideError(errorDiv);
    
    const formData = new FormData(e.target);
    const totpData = {
        token: formData.get('totpCode')
    };
    
    try {
        const response = await fetch('/api/verify-totp-setup', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(totpData)
        });
        
        const result = await response.json();
        
        if (response.ok && result.success) {
            // Marquer la configuration comme terminée
            sessionStorage.setItem('totp_setup_complete', 'true');
            sessionStorage.removeItem('requires_2fa_setup');
            sessionStorage.setItem('authenticated', 'true');
            
            showStep(3);
        } else {
            showError(errorDiv, result.error || 'Code de vérification incorrect');
        }
    } catch (error) {
        console.error('Erreur lors de la vérification TOTP:', error);
        showError(errorDiv, 'Erreur de connexion au serveur');
    } finally {
        hideLoading(totpBtn, btnText, btnLoading);
    }
}

/**
 * Copie la clé secrète dans le presse-papiers
 */
async function copySecretKey() {
    try {
        await navigator.clipboard.writeText(secretKey);
        
        const button = document.getElementById('copySecret');
        const originalText = button.textContent;
        
        button.textContent = '✅ Copié !';
        button.style.backgroundColor = '#28a745';
        
        setTimeout(() => {
            button.textContent = originalText;
            button.style.backgroundColor = '';
        }, 2000);
    } catch (error) {
        console.error('Erreur lors de la copie:', error);
        // Fallback pour les navigateurs plus anciens
        const textField = document.createElement('textarea');
        textField.value = secretKey;
        document.body.appendChild(textField);
        textField.select();
        document.execCommand('copy');
        document.body.removeChild(textField);
        
        showNotification('Clé secrète copiée !', 'success');
    }
}

/**
 * Valide la robustesse d'un mot de passe
 */
function validatePasswordStrength(password) {
    const errors = [];

    // Minimum 10 caractères
    if (password.length < 10) {
        errors.push('• Au moins 10 caractères');
    }

    // Au moins une majuscule
    if (!/[A-Z]/.test(password)) {
        errors.push('• Au moins une lettre majuscule (A-Z)');
    }

    // Au moins une minuscule
    if (!/[a-z]/.test(password)) {
        errors.push('• Au moins une lettre minuscule (a-z)');
    }

    // Au moins un chiffre
    if (!/[0-9]/.test(password)) {
        errors.push('• Au moins un chiffre (0-9)');
    }

    // Au moins un caractère spécial
    if (!/[!@#$%^&*()_+\-=\[\]{};':"\\|,.<>\/?~`]/.test(password)) {
        errors.push('• Au moins un caractère spécial (!@#$%^&*()_+-=[]{}|;:,.<>?~)');
    }

    return {
        isValid: errors.length === 0,
        errors: errors
    };
}

/**
 * Affiche la force du mot de passe en temps réel
 */
function showPasswordStrength() {
    const password = document.getElementById('newPassword').value;
    const strengthDiv = document.getElementById('passwordStrength');

    if (!strengthDiv) return; // Si l'élément n'existe pas encore

    if (password.length === 0) {
        strengthDiv.style.display = 'none';
        return;
    }

    const validation = validatePasswordStrength(password);
    strengthDiv.style.display = 'block';

    if (validation.isValid) {
        strengthDiv.innerHTML = '<div class="password-strength-good">✅ Mot de passe robuste</div>';
    } else {
        const errorList = validation.errors.map(error => `<div class="password-requirement">${error}</div>`).join('');
        strengthDiv.innerHTML = `<div class="password-strength-weak">Exigences manquantes :</div>${errorList}`;
    }
}

/**
 * Valide la correspondance des mots de passe
 */
function validatePasswordMatch() {
    const newPassword = document.getElementById('newPassword').value;
    const confirmPassword = document.getElementById('confirmPassword').value;
    const confirmField = document.getElementById('confirmPassword');

    if (confirmPassword && newPassword !== confirmPassword) {
        confirmField.setCustomValidity('Les mots de passe ne correspondent pas');
    } else {
        confirmField.setCustomValidity('');
    }
}

/**
 * Formate le code TOTP (6 chiffres uniquement)
 */
function formatTOTPCode(e) {
    let value = e.target.value.replace(/\D/g, ''); // Supprimer tous les non-chiffres
    if (value.length > 6) {
        value = value.substring(0, 6);
    }
    e.target.value = value;
}

/**
 * Affiche un état de chargement
 */
function showLoading(button, textSpan, loadingSpan) {
    textSpan.style.display = 'none';
    loadingSpan.style.display = 'inline';
    button.disabled = true;
}

/**
 * Cache l'état de chargement
 */
function hideLoading(button, textSpan, loadingSpan) {
    textSpan.style.display = 'inline';
    loadingSpan.style.display = 'none';
    button.disabled = false;
}

/**
 * Affiche un message d'erreur
 */
function showError(errorDiv, message) {
    errorDiv.textContent = message;
    errorDiv.style.display = 'block';
}

/**
 * Cache un message d'erreur
 */
function hideError(errorDiv) {
    errorDiv.style.display = 'none';
}

/**
 * Affiche une notification temporaire
 */
function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.textContent = message;
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 12px 24px;
        border-radius: 8px;
        background: ${type === 'success' ? '#28a745' : '#007bff'};
        color: white;
        z-index: 10000;
        animation: slideIn 0.3s ease-out;
    `;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.style.animation = 'slideOut 0.3s ease-in';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }, 3000);
} 