const express = require('express');
const session = require('express-session');
const bcrypt = require('bcrypt');
const { v4: uuidv4 } = require('uuid');
const multer = require('multer');
const sharp = require('sharp');
const fs = require('fs').promises;
const path = require('path');
const crypto = require('crypto');
const speakeasy = require('speakeasy');
const QRCode = require('qrcode');

// Chargement des variables d'environnement
require('dotenv').config();

const app = express();
const PORT = process.env.PORT || 80;

// Configuration du cryptage
const ENCRYPTION_KEY = process.env.ENCRYPTION_KEY || 'portail-vert-default-key-2024-secure-conversations-json-data-protection';
const ALGORITHM = 'aes-256-cbc';

// Fonction pour générer une clé de cryptage à partir d'une chaîne
function deriveKey(password) {
    return crypto.scryptSync(password, 'portail-vert-salt', 32);
}

// Fonction pour générer HMAC pour l'authentification
function generateHMAC(data, key) {
    return crypto.createHmac('sha256', key).update(data).digest('hex');
}

// Fonction pour crypter des données JSON
function encryptData(data) {
    try {
        const jsonString = JSON.stringify(data);
        const key = deriveKey(ENCRYPTION_KEY);
        const iv = crypto.randomBytes(16);
        const cipher = crypto.createCipheriv(ALGORITHM, key, iv);
        
        let encrypted = cipher.update(jsonString, 'utf8', 'hex');
        encrypted += cipher.final('hex');
        
        // Générer HMAC pour l'authentification
        const hmac = generateHMAC(encrypted + iv.toString('hex'), key);
        
        return {
            encrypted: encrypted,
            iv: iv.toString('hex'),
            hmac: hmac
        };
    } catch (error) {
        console.error('Erreur lors du cryptage:', error);
        throw error;
    }
}

// Fonction pour décrypter des données JSON
function decryptData(encryptedObj) {
    try {
        const key = deriveKey(ENCRYPTION_KEY);
        
        // Vérifier l'authentification HMAC
        const expectedHMAC = generateHMAC(encryptedObj.encrypted + encryptedObj.iv, key);
        if (expectedHMAC !== encryptedObj.hmac) {
            throw new Error('Authentification HMAC échouée - données corrompues ou clé incorrecte');
        }
        
        const iv = Buffer.from(encryptedObj.iv, 'hex');
        const decipher = crypto.createDecipheriv(ALGORITHM, key, iv);
        
        let decrypted = decipher.update(encryptedObj.encrypted, 'hex', 'utf8');
        decrypted += decipher.final('utf8');
        
        return JSON.parse(decrypted);
    } catch (error) {
        console.error('Erreur lors du décryptage:', error);
        throw error;
    }
}

// Fonction pour vérifier si un fichier est crypté
function isEncryptedFile(data) {
    return typeof data === 'object' && 
           data.encrypted && 
           data.iv && 
           (data.hmac || data.authTag) && 
           !Array.isArray(data);
}

// Fonction pour lire les gestionnaires
async function readManagers() {
    try {
        const data = await fs.readFile(path.join(__dirname, 'managers.json'), 'utf8');
        const parsedData = JSON.parse(data);
        
        // Vérifier si le fichier est crypté
        if (isEncryptedFile(parsedData)) {
            console.log('🔐 Décryptage du fichier gestionnaires...');
            return decryptData(parsedData);
        }
        
        return parsedData;
    } catch (error) {
        console.error('Erreur lors de la lecture des gestionnaires:', error);
        return { managers: [] };
    }
}

// Fonction pour écrire les gestionnaires
async function writeManagers(managersData) {
    try {
        console.log('🔐 Cryptage du fichier gestionnaires...');
        const encryptedData = encryptData(managersData);
        await fs.writeFile(path.join(__dirname, 'managers.json'), JSON.stringify(encryptedData, null, 2));
        return true;
    } catch (error) {
        console.error('Erreur lors de l\'écriture des gestionnaires:', error);
        return false;
    }
}

// Fonction pour trouver un gestionnaire
async function findManager(employeeNumber) {
    const managersData = await readManagers();
    return managersData.managers.find(m => m.employeeNumber === employeeNumber);
}

// Fonction pour valider la robustesse d'un mot de passe
function validatePasswordStrength(password) {
    const errors = [];

    // Minimum 10 caractères
    if (password.length < 10) {
        errors.push('Le mot de passe doit contenir au moins 10 caractères');
    }

    // Au moins une majuscule
    if (!/[A-Z]/.test(password)) {
        errors.push('Le mot de passe doit contenir au moins une lettre majuscule');
    }

    // Au moins une minuscule
    if (!/[a-z]/.test(password)) {
        errors.push('Le mot de passe doit contenir au moins une lettre minuscule');
    }

    // Au moins un chiffre
    if (!/[0-9]/.test(password)) {
        errors.push('Le mot de passe doit contenir au moins un chiffre');
    }

    // Au moins un caractère spécial
    if (!/[!@#$%^&*()_+\-=\[\]{};':"\\|,.<>\/?~`]/.test(password)) {
        errors.push('Le mot de passe doit contenir au moins un caractère spécial (!@#$%^&*()_+-=[]{}|;:,.<>?~)');
    }

    return {
        isValid: errors.length === 0,
        errors: errors
    };
}

// Middleware de base
app.use(express.json());
app.use(express.urlencoded({ extended: true }));
app.use(express.static('public'));

// Configuration des sessions
app.use(session({
    secret: 'portail-vert-secret-key-2024',
    resave: false,
    saveUninitialized: false,
    cookie: {
        secure: process.env.NODE_ENV === "production", // HTTPS en production // Mettre à true en production avec HTTPS
        maxAge: 24 * 60 * 60 * 1000 // 24 heures
    }
}));

// Configuration de multer pour l'upload d'images
const storage = multer.memoryStorage();
const upload = multer({
    storage: storage,
    limits: {
        fileSize: 20 * 1024 * 1024 // 20MB max
    },
    fileFilter: (req, file, cb) => {
        if (file.mimetype.startsWith('image/')) {
            cb(null, true);
        } else {
            cb(new Error('Seules les images sont autorisées'), false);
        }
    }
});

// Middleware de vérification d'authentification
const requireAuth = (req, res, next) => {
    if (req.session && req.session.authenticated) {
        next();
    } else {
        res.status(401).json({ error: 'Non autorisé' });
    }
};

// Fonction utilitaire pour lire un fichier de session
async function readSessionFile(sessionId) {
    try {
        const filePath = path.join(__dirname, 'sessions', `${sessionId}.json`);
        const data = await fs.readFile(filePath, 'utf8');
        const parsedData = JSON.parse(data);
        
        // Vérifier si le fichier est crypté
        if (isEncryptedFile(parsedData)) {
            return decryptData(parsedData);
        }
        
        return parsedData;
    } catch (error) {
        return null;
    }
}

// Fonction utilitaire pour écrire un fichier de session
async function writeSessionFile(sessionId, data) {
    try {
        const filePath = path.join(__dirname, 'sessions', `${sessionId}.json`);
        const encryptedData = encryptData(data);
        await fs.writeFile(filePath, JSON.stringify(encryptedData, null, 2));
        return true;
    } catch (error) {
        console.error('Erreur lors de l\'écriture du fichier de session:', error);
        return false;
    }
}

// Fonction pour détecter les numéros de téléphone dans une conversation
function detectPhoneNumbers(sessionData) {
    console.log('🔍 Début de la détection de numéros de téléphone...');
    
    // Patterns pour détecter les numéros de téléphone (formats couramment utilisés)
    const phonePatterns = [
        /\b\d{3}-\d{3}-\d{4}\b/g,           // Format 555-555-5555
        /\b\d{10}\b/g,                      // Format 5555555555 (10 chiffres consécutifs)
        /\b\(\d{3}\)\s?\d{3}-\d{4}\b/g,     // Format (555) 555-5555
        /\b\d{3}\.\d{3}\.\d{4}\b/g,         // Format 555.555.5555
        /\b\d{3}\s\d{3}\s\d{4}\b/g,         // Format 555 555 5555
        /\b1-\d{3}-\d{3}-\d{4}\b/g,         // Format 1-555-555-5555
        /\b\+1\s?\d{3}\s?\d{3}\s?\d{4}\b/g  // Format +1 555 555 5555
    ];

    const foundNumbers = [];
    let messagesAnalyzed = 0;

    // Vérifier tous les messages texte
    if (sessionData.messages && Array.isArray(sessionData.messages)) {
        for (const message of sessionData.messages) {
            if (message.type === 'text' && message.content) {
                messagesAnalyzed++;
                console.log(`🔍 Analyse du message ${messagesAnalyzed}: "${message.content.substring(0, 50)}..."`);
                
                for (const pattern of phonePatterns) {
                    const matches = message.content.match(pattern);
                    if (matches) {
                        console.log(`📞 Numéro(s) détecté(s) avec pattern ${pattern}:`, matches);
                        foundNumbers.push(...matches);
                    }
                }
            }
        }
    }

    console.log(`🔍 Analyse terminée: ${messagesAnalyzed} messages analysés, ${foundNumbers.length} numéros détectés`);
    if (foundNumbers.length > 0) {
        console.log('📞 Numéros détectés:', foundNumbers);
    }

    return foundNumbers.length > 0 ? [...new Set(foundNumbers)] : null; // Retourner les numéros uniques ou null
}

// Fonction pour supprimer complètement une conversation (incluant les photos)
async function deleteConversationCompletely(sessionId, sessionData) {
    try {
        // Supprimer toutes les images associées à cette conversation
        if (sessionData.messages) {
            for (const message of sessionData.messages) {
                if (message.type === 'image' && message.imagePath) {
                    try {
                        const fullImagePath = path.join(__dirname, message.imagePath);
                        await fs.unlink(fullImagePath);
                        console.log(`🗑️ Image supprimée: ${message.imagePath}`);
                    } catch (error) {
                        console.warn(`⚠️ Impossible de supprimer l'image ${message.imagePath}:`, error.message);
                    }
                }
            }
        }

        // Supprimer le fichier de session seulement s'il existe
        const sessionPath = path.join(__dirname, 'sessions', `${sessionId}.json`);
        try {
            await fs.unlink(sessionPath);
            console.log(`🗑️ Conversation ${sessionId} supprimée complètement (incluant ${sessionData.messages?.filter(m => m.type === 'image').length || 0} images)`);
        } catch (error) {
            if (error.code === 'ENOENT') {
                console.log(`⚠️ Fichier de session ${sessionId} déjà supprimé`);
            } else {
                throw error; // Re-lancer l'erreur si ce n'est pas un fichier manquant
            }
        }

        return true;
    } catch (error) {
        console.error('Erreur lors de la suppression complète:', error);
        return false;
    }
}

// Fonction pour archiver une conversation
async function archiveConversation(sessionId, sessionData) {
    try {
        // Créer le dossier archives s'il n'existe pas
        const archivesDir = path.join(__dirname, 'archives');
        try {
            await fs.mkdir(archivesDir, { recursive: true });
        } catch (error) {
            // Dossier existe déjà
        }

        // Créer un dossier par date
        const date = new Date().toISOString().split('T')[0]; // YYYY-MM-DD
        const dailyArchiveDir = path.join(archivesDir, date);
        try {
            await fs.mkdir(dailyArchiveDir, { recursive: true });
        } catch (error) {
            // Dossier existe déjà
        }

        // Copier le fichier de session vers les archives (crypté)
        const archivePath = path.join(dailyArchiveDir, `${sessionId}.json`);
        const encryptedData = encryptData(sessionData);
        await fs.writeFile(archivePath, JSON.stringify(encryptedData, null, 2));

        console.log(`📁 Conversation ${sessionId} archivée dans ${archivePath}`);
        return true;
    } catch (error) {
        console.error('Erreur lors de l\'archivage:', error);
        return false;
    }
}

// Routes d'authentification avec 2FA
app.post('/api/login', async (req, res) => {
    const { username, password } = req.body;

    try {
        const manager = await findManager(username);

        if (manager) {
            const isValid = await bcrypt.compare(password, manager.passwordHash);
            if (isValid) {
                // Stocker les informations temporaires dans la session
                req.session.tempAuth = {
                    managerId: manager.id,
                    username: username,
                    role: manager.role,
                    timestamp: Date.now()
                };

                // Vérifier si c'est la première connexion ou si 2FA n'est pas configuré
                if (manager.isFirstLogin || !manager.totpSecret) {
                    req.session.tempAuth.requiresSetup = true;
                    return res.json({ 
                        success: true, 
                        requiresSetup: true,
                        redirect: '/setup-2fa.html' 
                    });
                }

                // Rediriger vers la vérification 2FA
                res.json({ 
                    success: true, 
                    requires2FA: true,
                    redirect: '/login-2fa.html' 
                });
            } else {
                res.status(401).json({ error: 'Identifiants incorrects' });
            }
        } else {
            res.status(401).json({ error: 'Identifiants incorrects' });
        }
    } catch (error) {
        console.error('Erreur lors de la connexion:', error);
        res.status(500).json({ error: 'Erreur serveur' });
    }
});

// Vérification TOTP pour connexion
app.post('/api/verify-totp-login', async (req, res) => {
    const { token } = req.body;

    try {
        // Vérifier la session temporaire
        if (!req.session.tempAuth || !req.session.tempAuth.managerId) {
            return res.status(401).json({ error: 'Session expirée, veuillez vous reconnecter' });
        }

        // Vérifier le délai d'expiration (5 minutes)
        const currentTime = Date.now();
        const sessionTime = req.session.tempAuth.timestamp;
        if (currentTime - sessionTime > 5 * 60 * 1000) {
            delete req.session.tempAuth;
            return res.status(401).json({ error: 'Session expirée, veuillez vous reconnecter' });
        }

        const manager = await findManager(req.session.tempAuth.username);
        if (!manager || !manager.totpSecret) {
            return res.status(401).json({ error: 'Configuration 2FA manquante' });
        }

        // Vérifier le token TOTP
        const verified = speakeasy.totp.verify({
            secret: manager.totpSecret,
            encoding: 'base32',
            token: token,
            window: 2 // Permettre une fenêtre de +/- 2 intervalles (60 secondes)
        });

        if (verified) {
            // Authentification complète réussie
            req.session.authenticated = true;
            req.session.username = req.session.tempAuth.username;
            req.session.managerId = req.session.tempAuth.managerId;
            req.session.role = req.session.tempAuth.role;

            // Nettoyer la session temporaire
            delete req.session.tempAuth;

            // Mettre à jour la dernière connexion
            const managersData = await readManagers();
            const managerIndex = managersData.managers.findIndex(m => m.id === manager.id);
            if (managerIndex !== -1) {
                managersData.managers[managerIndex].lastLogin = new Date().toISOString();
                await writeManagers(managersData);
            }

            res.json({ success: true, redirect: '/dashboard.html' });
        } else {
            res.status(401).json({ error: 'Code de vérification incorrect' });
        }
    } catch (error) {
        console.error('Erreur lors de la vérification TOTP:', error);
        res.status(500).json({ error: 'Erreur serveur' });
    }
});

// Configuration 2FA - Changement de mot de passe initial
app.post('/api/change-password-2fa', async (req, res) => {
    const { currentPassword, newPassword } = req.body;

    try {
        if (!req.session.tempAuth || !req.session.tempAuth.requiresSetup) {
            return res.status(401).json({ error: 'Accès non autorisé' });
        }

        const manager = await findManager(req.session.tempAuth.username);
        if (!manager) {
            return res.status(401).json({ error: 'Gestionnaire non trouvé' });
        }

        // Vérifier le mot de passe actuel
        const isValid = await bcrypt.compare(currentPassword, manager.passwordHash);
        if (!isValid) {
            return res.status(401).json({ error: 'Mot de passe actuel incorrect' });
        }

        // Validation du nouveau mot de passe
        const passwordValidation = validatePasswordStrength(newPassword);
        if (!passwordValidation.isValid) {
            return res.status(400).json({
                error: 'Mot de passe non conforme aux exigences de sécurité',
                details: passwordValidation.errors
            });
        }

        // Hacher le nouveau mot de passe
        const newPasswordHash = await bcrypt.hash(newPassword, 10);

        // Mettre à jour le gestionnaire
        const managersData = await readManagers();
        const managerIndex = managersData.managers.findIndex(m => m.id === manager.id);
        if (managerIndex !== -1) {
            managersData.managers[managerIndex].passwordHash = newPasswordHash;
            managersData.managers[managerIndex].isFirstLogin = false;
            const success = await writeManagers(managersData);
            
            if (success) {
                res.json({ success: true });
            } else {
                res.status(500).json({ error: 'Erreur lors de la sauvegarde' });
            }
        } else {
            res.status(404).json({ error: 'Gestionnaire non trouvé' });
        }
    } catch (error) {
        console.error('Erreur lors du changement de mot de passe:', error);
        res.status(500).json({ error: 'Erreur serveur' });
    }
});

// Génération du secret TOTP
app.post('/api/generate-totp-secret', async (req, res) => {
    try {
        if (!req.session.tempAuth || !req.session.tempAuth.requiresSetup) {
            return res.status(401).json({ error: 'Accès non autorisé' });
        }

        const manager = await findManager(req.session.tempAuth.username);
        if (!manager) {
            return res.status(401).json({ error: 'Gestionnaire non trouvé' });
        }

        // Générer un secret TOTP
        const secret = speakeasy.generateSecret({
            name: `Portail Vert (${manager.name})`,
            issuer: 'Portail Vert'
        });

        // Stocker le secret temporairement dans la session
        req.session.tempAuth.tempTotpSecret = secret.base32;

        // Générer le QR code
        const qrCodeDataURL = await QRCode.toDataURL(secret.otpauth_url);

        res.json({
            success: true,
            secret: secret.base32,
            qrCodeUrl: qrCodeDataURL
        });
    } catch (error) {
        console.error('Erreur lors de la génération du secret TOTP:', error);
        res.status(500).json({ error: 'Erreur serveur' });
    }
});

// Vérification TOTP pour configuration initiale
app.post('/api/verify-totp-setup', async (req, res) => {
    const { token } = req.body;

    try {
        if (!req.session.tempAuth || !req.session.tempAuth.requiresSetup || !req.session.tempAuth.tempTotpSecret) {
            return res.status(401).json({ error: 'Configuration 2FA non initialisée' });
        }

        // Vérifier le token TOTP avec le secret temporaire
        const verified = speakeasy.totp.verify({
            secret: req.session.tempAuth.tempTotpSecret,
            encoding: 'base32',
            token: token,
            window: 2
        });

        if (verified) {
            // Sauvegarder le secret TOTP pour le gestionnaire
            const managersData = await readManagers();
            const managerIndex = managersData.managers.findIndex(m => m.id === req.session.tempAuth.managerId);
            
            if (managerIndex !== -1) {
                managersData.managers[managerIndex].totpSecret = req.session.tempAuth.tempTotpSecret;
                managersData.managers[managerIndex].totpEnabled = true;
                managersData.managers[managerIndex].isFirstLogin = false;
                
                const success = await writeManagers(managersData);
                if (success) {
                    // Configuration terminée, authentifier complètement
                    req.session.authenticated = true;
                    req.session.username = req.session.tempAuth.username;
                    req.session.managerId = req.session.tempAuth.managerId;
                    req.session.role = req.session.tempAuth.role;
                    
                    // Nettoyer la session temporaire
                    delete req.session.tempAuth;
                    
                    res.json({ success: true, redirect: '/dashboard.html' });
                } else {
                    res.status(500).json({ error: 'Erreur lors de la sauvegarde de la configuration 2FA' });
                }
            } else {
                res.status(404).json({ error: 'Gestionnaire non trouvé' });
            }
        } else {
            res.status(401).json({ error: 'Code de vérification incorrect' });
        }
    } catch (error) {
        console.error('Erreur lors de la vérification TOTP setup:', error);
        res.status(500).json({ error: 'Erreur serveur' });
    }
});

app.post('/api/logout', (req, res) => {
    req.session.destroy();
    res.json({ success: true });
});

// Middleware pour vérifier les droits admin
const requireAdmin = (req, res, next) => {
    if (req.session && req.session.authenticated && req.session.role === 'admin') {
        next();
    } else {
        res.status(403).json({ error: 'Accès administrateur requis' });
    }
};

// Routes d'administration des gestionnaires
app.get('/admin/managers', requireAdmin, async (req, res) => {
    try {
        const managersData = await readManagers();
        // Ne pas renvoyer les mots de passe et secrets TOTP
        const safeManagers = managersData.managers.map(m => ({
            id: m.id,
            name: m.name,
            employeeNumber: m.employeeNumber,
            role: m.role,
            createdAt: m.createdAt,
            lastLogin: m.lastLogin,
            isFirstLogin: m.isFirstLogin || false,
            totpEnabled: m.totpEnabled || false
        }));
        res.json({ managers: safeManagers });
    } catch (error) {
        console.error('Erreur lors de la récupération des gestionnaires:', error);
        res.status(500).json({ error: 'Erreur serveur' });
    }
});

app.post('/admin/managers', requireAdmin, async (req, res) => {
    try {
        const { name, employeeNumber, password, role = 'manager' } = req.body;

        if (!name || !employeeNumber || !password) {
            return res.status(400).json({ error: 'Tous les champs sont requis' });
        }

        const managersData = await readManagers();

        // Vérifier si le numéro d'employé existe déjà
        if (managersData.managers.find(m => m.employeeNumber === employeeNumber)) {
            return res.status(400).json({ error: 'Ce numéro d\'employé existe déjà' });
        }

        // Hacher le mot de passe
        const passwordHash = await bcrypt.hash(password, 10);

        // Créer le nouveau gestionnaire
        const newManager = {
            id: uuidv4(),
            name,
            employeeNumber,
            passwordHash,
            role,
            createdAt: new Date().toISOString(),
            lastLogin: null,
            isFirstLogin: true, // Forcer la configuration 2FA
            totpSecret: null,
            totpEnabled: false
        };

        managersData.managers.push(newManager);

        const success = await writeManagers(managersData);
        if (success) {
            res.json({ success: true, manager: {
                id: newManager.id,
                name: newManager.name,
                employeeNumber: newManager.employeeNumber,
                role: newManager.role,
                createdAt: newManager.createdAt,
                lastLogin: newManager.lastLogin
            }});
        } else {
            res.status(500).json({ error: 'Erreur lors de la sauvegarde' });
        }
    } catch (error) {
        console.error('Erreur lors de la création du gestionnaire:', error);
        res.status(500).json({ error: 'Erreur serveur' });
    }
});

app.put('/admin/managers/:id', requireAdmin, async (req, res) => {
    try {
        const { id } = req.params;
        const { name, employeeNumber, password, role } = req.body;

        const managersData = await readManagers();
        const managerIndex = managersData.managers.findIndex(m => m.id === id);

        if (managerIndex === -1) {
            return res.status(404).json({ error: 'Gestionnaire non trouvé' });
        }

        // Vérifier si le numéro d'employé est déjà utilisé par un autre gestionnaire
        const existingManager = managersData.managers.find(m => m.employeeNumber === employeeNumber && m.id !== id);
        if (existingManager) {
            return res.status(400).json({ error: 'Ce numéro d\'employé est déjà utilisé' });
        }

        // Mettre à jour les champs
        if (name) managersData.managers[managerIndex].name = name;
        if (employeeNumber) managersData.managers[managerIndex].employeeNumber = employeeNumber;
        if (role) managersData.managers[managerIndex].role = role;

        // Mettre à jour le mot de passe si fourni
        if (password) {
            managersData.managers[managerIndex].passwordHash = await bcrypt.hash(password, 10);
        }

        const success = await writeManagers(managersData);
        if (success) {
            const updatedManager = managersData.managers[managerIndex];
            res.json({ success: true, manager: {
                id: updatedManager.id,
                name: updatedManager.name,
                employeeNumber: updatedManager.employeeNumber,
                role: updatedManager.role,
                createdAt: updatedManager.createdAt,
                lastLogin: updatedManager.lastLogin
            }});
        } else {
            res.status(500).json({ error: 'Erreur lors de la sauvegarde' });
        }
    } catch (error) {
        console.error('Erreur lors de la modification du gestionnaire:', error);
        res.status(500).json({ error: 'Erreur serveur' });
    }
});

app.delete('/admin/managers/:id', requireAdmin, async (req, res) => {
    try {
        const { id } = req.params;

        const managersData = await readManagers();
        const managerIndex = managersData.managers.findIndex(m => m.id === id);

        if (managerIndex === -1) {
            return res.status(404).json({ error: 'Gestionnaire non trouvé' });
        }

        // Empêcher la suppression du dernier admin
        const adminCount = managersData.managers.filter(m => m.role === 'admin').length;
        if (managersData.managers[managerIndex].role === 'admin' && adminCount <= 1) {
            return res.status(400).json({ error: 'Impossible de supprimer le dernier administrateur' });
        }

        managersData.managers.splice(managerIndex, 1);

        const success = await writeManagers(managersData);
        if (success) {
            res.json({ success: true });
        } else {
            res.status(500).json({ error: 'Erreur lors de la sauvegarde' });
        }
    } catch (error) {
        console.error('Erreur lors de la suppression du gestionnaire:', error);
        res.status(500).json({ error: 'Erreur serveur' });
    }
});

// Réinitialisation TOTP pour un gestionnaire
app.post('/admin/managers/:id/reset-totp', requireAdmin, async (req, res) => {
    try {
        const { id } = req.params;

        const managersData = await readManagers();
        const managerIndex = managersData.managers.findIndex(m => m.id === id);

        if (managerIndex === -1) {
            return res.status(404).json({ error: 'Gestionnaire non trouvé' });
        }

        // Réinitialiser le TOTP
        managersData.managers[managerIndex].totpSecret = null;
        managersData.managers[managerIndex].totpEnabled = false;
        managersData.managers[managerIndex].isFirstLogin = true; // Forcer la reconfiguration

        const success = await writeManagers(managersData);
        if (success) {
            res.json({ 
                success: true, 
                message: `TOTP réinitialisé pour ${managersData.managers[managerIndex].name}. L'utilisateur devra reconfigurer son 2FA à la prochaine connexion.` 
            });
        } else {
            res.status(500).json({ error: 'Erreur lors de la sauvegarde' });
        }
    } catch (error) {
        console.error('Erreur lors de la réinitialisation TOTP:', error);
        res.status(500).json({ error: 'Erreur serveur' });
    }
});

// Fonction pour générer le lien de chat approprié
// Fonction pour générer les liens de chat
function generateChatLink(req, sessionId) {
    const protocol = req.secure || req.headers['x-forwarded-proto'] === 'https' ? 'https' : 'http';
    const host = req.get('host');
    
    // En production sur mathix.xyz
    if (process.env.NODE_ENV === 'production' || host.includes('mathix.xyz')) {
        return `https://mathix.xyz/chat.html?session=${sessionId}`;
    }
    
    // Pour les tests locaux
    return `${protocol}://${host}/chat.html?session=${sessionId}`;
}

// Routes du portail gestionnaire
app.post('/portal/create-chat', requireAuth, async (req, res) => {
    const sessionId = uuidv4();
    const sessionData = {
        id: sessionId,
        messages: [],
        technicienConnecte: false,
        createdAt: new Date().toISOString(),
        lastActivity: new Date().toISOString()
    };

    const success = await writeSessionFile(sessionId, sessionData);
    if (success) {
        const chatLink = generateChatLink(req, sessionId);
        res.json({
            success: true,
            sessionId: sessionId,
            chatLink: chatLink
        });
    } else {
        res.status(500).json({ error: 'Erreur lors de la création de la session' });
    }
});

app.get('/portal/conversations', requireAuth, async (req, res) => {
    try {
        const sessionsDir = path.join(__dirname, 'sessions');
        const files = await fs.readdir(sessionsDir);
        const conversations = [];
        
        for (const file of files) {
            if (file.endsWith('.json')) {
                const sessionId = file.replace('.json', '');
                const sessionData = await readSessionFile(sessionId);
                if (sessionData && sessionData.status !== 'terminated') { // Exclure les terminées
                    conversations.push({
                        id: sessionId,
                        createdAt: sessionData.createdAt,
                        lastActivity: sessionData.lastActivity,
                        messageCount: sessionData.messages.length,
                        technicienConnecte: sessionData.technicienConnecte,
                        status: sessionData.status || 'active'
                    });
                }
            }
        }
        
        // Trier par dernière activité
        conversations.sort((a, b) => new Date(b.lastActivity) - new Date(a.lastActivity));
        res.json(conversations);
    } catch (error) {
        console.error('Erreur lors de la récupération des conversations:', error);
        res.status(500).json({ error: 'Erreur serveur' });
    }
});

// Route pour récupérer les conversations archivées
app.get('/portal/archives', requireAuth, async (req, res) => {
    try {
        const { date } = req.query;
        const archivesDir = path.join(__dirname, 'archives');
        const archives = [];
        
        if (date) {
            // Récupérer les archives d'une date spécifique
            const dailyArchiveDir = path.join(archivesDir, date);
            try {
                const files = await fs.readdir(dailyArchiveDir);
                for (const file of files) {
                    if (file.endsWith('.json')) {
                        const sessionId = file.replace('.json', '');
                        const archivePath = path.join(dailyArchiveDir, file);
                        const rawData = JSON.parse(await fs.readFile(archivePath, 'utf8'));
                        const archiveData = isEncryptedFile(rawData) ? decryptData(rawData) : rawData;
                        
                        archives.push({
                            id: sessionId,
                            createdAt: archiveData.createdAt,
                            lastActivity: archiveData.lastActivity,
                            terminatedAt: archiveData.terminatedAt,
                            terminatedBy: archiveData.terminatedBy,
                            messageCount: archiveData.messages.length,
                            date: date
                        });
                    }
                }
            } catch (error) {
                // Pas d'archives pour cette date
            }
        } else {
            // Récupérer la liste des dates disponibles
            try {
                const dates = await fs.readdir(archivesDir);
                for (const dateDir of dates) {
                    const dailyArchiveDir = path.join(archivesDir, dateDir);
                    const stat = await fs.stat(dailyArchiveDir);
                    if (stat.isDirectory()) {
                        const files = await fs.readdir(dailyArchiveDir);
                        const conversationCount = files.filter(f => f.endsWith('.json')).length;
                        if (conversationCount > 0) {
                            archives.push({
                                date: dateDir,
                                conversationCount: conversationCount
                            });
                        }
                    }
                }
                // Trier par date décroissante
                archives.sort((a, b) => b.date.localeCompare(a.date));
            } catch (error) {
                // Pas de dossier archives
            }
        }
        
        res.json(archives);
    } catch (error) {
        console.error('Erreur lors de la récupération des archives:', error);
        res.status(500).json({ error: 'Erreur serveur' });
    }
});

// Route pour récupérer le contenu complet d'une conversation archivée
app.get('/portal/archives/:date/:sessionId', requireAuth, async (req, res) => {
    try {
        const { date, sessionId } = req.params;
        const archivePath = path.join(__dirname, 'archives', date, `${sessionId}.json`);
        
        try {
            const rawData = JSON.parse(await fs.readFile(archivePath, 'utf8'));
            const archiveData = isEncryptedFile(rawData) ? decryptData(rawData) : rawData;
            res.json(archiveData);
        } catch (error) {
            if (error.code === 'ENOENT') {
                res.status(404).json({ error: 'Conversation archivée non trouvée' });
            } else {
                throw error;
            }
        }
    } catch (error) {
        console.error('Erreur lors de la récupération de la conversation archivée:', error);
        res.status(500).json({ error: 'Erreur serveur' });
    }
});

// Variable pour tracker les sessions en cours de terminaison (éviter les doubles appels)
const terminatingSessions = new Set();

app.delete('/portal/conversations/:sessionId', requireAuth, async (req, res) => {
    try {
        const { sessionId } = req.params;
        const { hasSensitiveData } = req.body; // Nouvelle donnée envoyée par le client

        // Vérifier si cette session est déjà en cours de terminaison
        if (terminatingSessions.has(sessionId)) {
            console.log(`⚠️ Session ${sessionId} déjà en cours de terminaison, ignorer la demande`);
            return res.json({ 
                success: true, 
                message: 'Traitement de la terminaison déjà en cours...', 
                action: 'processing' 
            });
        }

        // Marquer la session comme étant en cours de terminaison
        terminatingSessions.add(sessionId);

        console.log(`🔄 Terminaison de session ${sessionId}, données sensibles: ${hasSensitiveData}`);

        const sessionData = await readSessionFile(sessionId);

        if (!sessionData) {
            console.log(`❌ Session ${sessionId} non trouvée`);
            terminatingSessions.delete(sessionId); // Nettoyer le tracker
            return res.status(404).json({ error: 'Session non trouvée' });
        }

        // Vérifier si la session n'est pas déjà terminée
        if (sessionData.status === 'terminated') {
            console.log(`⚠️ Session ${sessionId} déjà terminée`);
            terminatingSessions.delete(sessionId); // Nettoyer le tracker
            return res.status(400).json({ error: 'Cette session a déjà été terminée' });
        }

        if (hasSensitiveData === true) {
            // Le gestionnaire a confirmé la présence de données sensibles
            // Supprimer complètement la conversation (incluant les photos)
            const deleteSuccess = await deleteConversationCompletely(sessionId, sessionData);
            
            // Nettoyer le tracker
            terminatingSessions.delete(sessionId);

            if (deleteSuccess) {
                res.json({
                    success: true,
                    message: 'Conversation supprimée complètement en raison de données sensibles',
                    action: 'deleted'
                });
            } else {
                res.status(500).json({ error: 'Erreur lors de la suppression complète de la conversation' });
            }
        } else {
            // Le gestionnaire a dit qu'il n'y a pas de données sensibles
            // Mais on vérifie quand même pour les numéros de téléphone
            const detectedPhones = detectPhoneNumbers(sessionData);

            if (detectedPhones) {
                // Des numéros de téléphone ont été détectés malgré la réponse du gestionnaire
                // Supprimer complètement la conversation
                const deleteSuccess = await deleteConversationCompletely(sessionId, sessionData);
                
                // Nettoyer le tracker
                terminatingSessions.delete(sessionId);

                if (deleteSuccess) {
                    res.json({
                        success: false,
                        error: 'Conversation non archivée : des informations sensibles ont été détectées automatiquement (numéros de téléphone). La conversation a été supprimée complètement.',
                        detectedPhones: detectedPhones,
                        action: 'deleted_auto'
                    });
                } else {
                    res.status(500).json({ error: 'Erreur lors de la suppression automatique de la conversation' });
                }
            } else {
                // Aucune donnée sensible détectée, procéder à l'archivage normal
                // Marquer comme terminée avant d'archiver
                sessionData.status = 'terminated';
                sessionData.terminatedAt = new Date().toISOString();
                sessionData.terminatedBy = req.session.username || 'gestionnaire';

                // Archiver la conversation
                const archiveSuccess = await archiveConversation(sessionId, sessionData);
                
                // Nettoyer le tracker
                terminatingSessions.delete(sessionId);
                
                if (archiveSuccess) {
                    // Supprimer le fichier de session original après archivage réussi
                    try {
                        const sessionPath = path.join(__dirname, 'sessions', `${sessionId}.json`);
                        await fs.unlink(sessionPath);
                        console.log(`📁 Session ${sessionId} archivée et fichier original supprimé`);
                    } catch (error) {
                        console.warn(`⚠️ Impossible de supprimer le fichier de session original ${sessionId}:`, error.message);
                    }

                    res.json({
                        success: true,
                        message: 'Session terminée et archivée avec succès',
                        action: 'archived'
                    });
                } else {
                    res.status(500).json({ error: 'Erreur lors de l\'archivage de la session' });
                }
            }
        }
    } catch (error) {
        console.error('Erreur lors de la terminaison de la session:', error);
        // Nettoyer le tracker en cas d'erreur
        const { sessionId } = req.params;
        terminatingSessions.delete(sessionId);
        res.status(500).json({ error: 'Erreur serveur lors de la terminaison' });
    }
});

// Routes de l'API de chat
app.get('/api/chat/:sessionId/connect', async (req, res) => {
    const { sessionId } = req.params;
    const sessionData = await readSessionFile(sessionId);
    
    if (!sessionData) {
        return res.status(404).json({ error: 'Session non trouvée' });
    }
    
    // Vérifier si la conversation est terminée
    if (sessionData.status === 'terminated') {
        return res.status(410).json({ 
            error: 'Conversation terminée', 
            terminated: true,
            terminatedAt: sessionData.terminatedAt,
            terminatedBy: sessionData.terminatedBy
        });
    }
    
    // Vérifier si un technicien est déjà connecté
    if (sessionData.technicienConnecte && !req.session.authenticated) {
        return res.status(403).json({ error: 'Un technicien est déjà connecté à cette session' });
    }
    
    // Marquer le technicien comme connecté si ce n'est pas le gestionnaire
    if (!req.session.authenticated) {
        sessionData.technicienConnecte = true;
        sessionData.lastActivity = new Date().toISOString();
        await writeSessionFile(sessionId, sessionData);
    }
    
    res.json({ success: true, canConnect: true });
});

app.get('/api/chat/:sessionId/messages', async (req, res) => {
    const { sessionId } = req.params;
    const { since } = req.query;

    const sessionData = await readSessionFile(sessionId);
    if (!sessionData) {
        return res.status(404).json({ error: 'Session non trouvée' });
    }

    let messages = sessionData.messages;

    // Filtrer les messages depuis le dernier ID reçu
    if (since) {
        const sinceIndex = messages.findIndex(msg => msg.id === since);
        if (sinceIndex !== -1) {
            messages = messages.slice(sinceIndex + 1);
        }
    }

    res.json({ messages });
});

app.post('/api/chat/:sessionId/message', async (req, res) => {
    const { sessionId } = req.params;
    const { message, sender } = req.body;

    const sessionData = await readSessionFile(sessionId);
    if (!sessionData) {
        return res.status(404).json({ error: 'Session non trouvée' });
    }

    // Vérifier si la conversation est terminée
    if (sessionData.status === 'terminated') {
        return res.status(410).json({ 
            error: 'Cette conversation a été terminée par le gestionnaire. Vous ne pouvez plus envoyer de messages.',
            terminated: true,
            terminatedAt: sessionData.terminatedAt,
            terminatedBy: sessionData.terminatedBy
        });
    }

    const newMessage = {
        id: uuidv4(),
        type: 'text',
        content: message,
        sender: sender || 'technicien',
        timestamp: new Date().toISOString()
    };

    sessionData.messages.push(newMessage);
    sessionData.lastActivity = new Date().toISOString();

    const success = await writeSessionFile(sessionId, sessionData);
    if (success) {
        res.json({ success: true, message: newMessage });
    } else {
        res.status(500).json({ error: 'Erreur lors de l\'envoi du message' });
    }
});

app.post('/api/chat/:sessionId/upload', upload.single('image'), async (req, res) => {
    const { sessionId } = req.params;
    const { sender } = req.body;

    if (!req.file) {
        return res.status(400).json({ error: 'Aucune image fournie' });
    }

    const sessionData = await readSessionFile(sessionId);
    if (!sessionData) {
        return res.status(404).json({ error: 'Session non trouvée' });
    }

    // Vérifier si la conversation est terminée
    if (sessionData.status === 'terminated') {
        return res.status(410).json({ 
            error: 'Cette conversation a été terminée par le gestionnaire. Vous ne pouvez plus envoyer d\'images.',
            terminated: true,
            terminatedAt: sessionData.terminatedAt,
            terminatedBy: sessionData.terminatedBy
        });
    }

    try {
        // Générer un nom de fichier unique
        const imageId = uuidv4();
        const filename = `${imageId}.jpg`;
        const filepath = path.join(__dirname, 'uploads', filename);

        // Compresser et redimensionner l'image avec Sharp
        await sharp(req.file.buffer)
            .resize(1280, null, {
                withoutEnlargement: true,
                fit: 'inside'
            })
            .jpeg({ quality: 75 })
            .toFile(filepath);

        // Créer le message image
        const imageMessage = {
            id: uuidv4(),
            type: 'image',
            content: `/uploads/${filename}`,
            sender: sender || 'technicien',
            timestamp: new Date().toISOString(),
            originalName: req.file.originalname
        };

        sessionData.messages.push(imageMessage);
        sessionData.lastActivity = new Date().toISOString();

        const success = await writeSessionFile(sessionId, sessionData);
        if (success) {
            res.json({ success: true, message: imageMessage });
        } else {
            res.status(500).json({ error: 'Erreur lors de la sauvegarde' });
        }
    } catch (error) {
        console.error('Erreur lors du traitement de l\'image:', error);
        res.status(500).json({ error: 'Erreur lors du traitement de l\'image' });
    }
});

// Route pour servir les images uploadées
app.use('/uploads', express.static('uploads'));

// Route pour vérifier l'état d'une session
app.get('/api/chat/:sessionId/status', async (req, res) => {
    const { sessionId } = req.params;
    const sessionData = await readSessionFile(sessionId);

    if (!sessionData) {
        return res.status(404).json({ error: 'Session non trouvée' });
    }

    res.json({
        exists: true,
        technicienConnecte: sessionData.technicienConnecte,
        messageCount: sessionData.messages.length,
        lastActivity: sessionData.lastActivity
    });
});

// Gestion des erreurs
app.use((error, req, res, next) => {
    if (error instanceof multer.MulterError) {
        if (error.code === 'LIMIT_FILE_SIZE') {
            return res.status(400).json({ error: 'Fichier trop volumineux (max 20MB)' });
        }
    }
    console.error('Erreur serveur:', error);
    res.status(500).json({ error: 'Erreur serveur interne' });
});

// Fonction pour obtenir l'adresse IP locale
function getLocalIPAddress() {
    const { networkInterfaces } = require('os');
    const nets = networkInterfaces();
    const results = [];

    for (const name of Object.keys(nets)) {
        for (const net of nets[name]) {
            // Ignorer les adresses non-IPv4 et internes (127.x.x.x)
            if (net.family === 'IPv4' && !net.internal) {
                results.push(net.address);
            }
        }
    }

    return results[0] || 'localhost';
}

// Démarrage du serveur sur toutes les interfaces (0.0.0.0)
app.listen(PORT, '0.0.0.0', () => {
    const localIP = getLocalIPAddress();

    console.log(`🌳 Portail Vert démarré sur le port ${PORT}`);
    console.log(`📱 Accès local (PC): http://localhost:${PORT}/login.html`);
    console.log(`📱 Accès réseau (téléphone): http://${localIP}:${PORT}/login.html`);
    console.log(`👤 Identifiants par défaut: admin / motdepassesecurise123`);
    console.log(`🔗 Pour tester le chat, utilisez: http://${localIP}:${PORT}/chat.html?session=ID`);
    console.log(`\n📋 Instructions de test:`);
    console.log(`   1. Connectez-vous sur PC: http://localhost:${PORT}/login.html`);
    console.log(`   2. Créez une session et copiez le lien`);
    console.log(`   3. Ouvrez le lien sur votre téléphone`);
    console.log(`   4. Testez l'échange de messages et photos`);
    console.log(`\n🔥 Serveur accessible sur le réseau local !`);
});
